/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.upgrade.tool.service;

import static com.mulesoft.runtime.upgrade.tool.service.utils.PathDefinitionUtils.getAllDefinedPaths;
import static com.mulesoft.runtime.upgrade.tool.utils.PrettyPrintingFormatter.formatAppliedPatches;
import static com.mulesoft.runtime.upgrade.tool.utils.PrettyPrintingFormatter.formatBackupDescriptor;
import static java.lang.String.format;

import com.mulesoft.runtime.upgrade.tool.domain.BackupDescriptor;
import com.mulesoft.runtime.upgrade.tool.domain.MuleDistribution;
import com.mulesoft.runtime.upgrade.tool.domain.SemVer;
import com.mulesoft.runtime.upgrade.tool.service.api.BackupService;
import com.mulesoft.runtime.upgrade.tool.service.api.DistroPatchService;
import com.mulesoft.runtime.upgrade.tool.service.api.FileSystemService;
import com.mulesoft.runtime.upgrade.tool.service.api.MuleStatusService;
import com.mulesoft.runtime.upgrade.tool.service.api.StatusPreconditionsValidatorService;
import com.mulesoft.runtime.upgrade.tool.service.api.UserInteractionService;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * A preconditions validator for the tool's main processes.
 */
@Service
public class DefaultStatusPreconditionsValidatorService implements StatusPreconditionsValidatorService {

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultStatusPreconditionsValidatorService.class);

  @Autowired
  private BackupService backupService;

  @Autowired
  private FileSystemService fileSystemService;

  @Autowired
  private MuleStatusService statusService;

  @Autowired
  private UserInteractionService userInteractionService;

  @Autowired
  private DistroPatchService patchService;

  @Autowired
  private DefaultPreconditionsValidatorService preconditionsValidatorService;

  private boolean getConsentForRemovingAppliedPatches(Path distroLocation, boolean force) throws IOException {
    List<File> appliedPatches = patchService.getAppliedPatches(distroLocation);

    if (appliedPatches.isEmpty()) {
      LOGGER.debug("No applied patches found for Mule Runtime distribution: [{}]", distroLocation);
      return true;
    } else {
      LOGGER.info("Applied patches found for Mule Runtime distribution: [{}]", distroLocation);
      String appliedPatchesListPrettyFormatted = format("%n%s", formatAppliedPatches(appliedPatches));
      LOGGER.info(appliedPatchesListPrettyFormatted);
      if (force) {
        LOGGER.debug("Consent to remove applied patches is set through force parameter.");
        return true;
      } else {
        return userInteractionService.confirmAction("remove patches", "Could patched be removed?", false);
      }
    }
  }

  /**
   * Checks upgrade preconditions for status command.
   *
   * @param oldMule {@link MuleDistribution} representing the old Mule Runtime distribution being upgraded.
   * @throws IOException if an I/O error occurs
   */
  @Override
  public List<String> checkUpgradeStatusOffline(MuleDistribution oldMule)
      throws IOException {

    LOGGER.debug("Checking preconditions status for upgrade process...");

    List<String> statusErrorMessages = new LinkedList<>();
    statusErrorMessages.add("Checking upgrade process preconditions...");

    statusErrorMessages.addAll(checkWritingPermissions(oldMule.getLocation()));
    statusErrorMessages.addAll(isMuleStopped(oldMule.getLocation()));

    return statusErrorMessages;
  }


  /**
   * Checks rollback preconditions for status command.
   *
   * @param muleDistroLocation {@link MuleDistribution} representing the current Mule Runtime distribution location.
   * @throws IOException if an I/O error occurs
   */
  @Override
  public List<String> checkRollbackPreconditions(Path muleDistroLocation) throws IOException {
    LOGGER.debug("Validating if all preconditions to proceed with the rollback are met...");

    List<String> statusErrorMessages = new LinkedList<>();

    statusErrorMessages.addAll(checkBackupExistence(muleDistroLocation));
    statusErrorMessages.addAll(preconditionsValidatorService.checkBackupIntegrityAndGetErrorMessage(muleDistroLocation));
    statusErrorMessages.addAll(checkWritingPermissions(muleDistroLocation));
    statusErrorMessages.addAll(isMuleStopped(muleDistroLocation));

    return statusErrorMessages;
  }

  private List<String> checkNewMuleIsNewer(SemVer oldDistro, SemVer newDistro) {
    List<String> messages = new LinkedList<>();
    if (oldDistro.compareTo(newDistro) >= 0) {
      String message = String.format("Version of new Mule distribution {%s} should be newer than old Mule distribution: {%s}",
                                     newDistro, oldDistro);
      messages.add(message);
      LOGGER.error(message);
    }
    return messages;
  }

  /**
   * Asks consent for overwriting the current BackUp.
   *
   * @param force If force is true, will not ask to the user.
   * @return User response.
   * @throws IOException I/O Exception.
   */
  public boolean getConsentForOverwritingBackup(boolean force) throws IOException {
    LOGGER.info("Asking for consent for overriding backup...");

    if (force) {
      LOGGER.debug("Consent to overwrite backup is set through force parameter.");
      return true;
    } else {
      return userInteractionService.confirmAction("overwrite backup", "Could backup be overwritten?", false);
    }
  }

  /**
   * Checks existence of any BackUp.
   *
   * @param muleDistroLocation Mule path to looking for the BackUp.
   * @return A list of Strings, if the list is empty, the BackUp exists, otherwise not exists.
   * @throws IOException I/O Exception.
   */
  public List<String> checkBackupExistence(Path muleDistroLocation) throws IOException {
    Optional<BackupDescriptor> backupDescriptor = backupService.readBackupDescriptor(muleDistroLocation);
    List<String> messages = new LinkedList<>();

    if (!backupDescriptor.isPresent()) {
      String message = String.format("Backup does not exist for Mule Runtime distribution: [%s].", muleDistroLocation);
      messages.add(message);
    } else {
      LOGGER.debug("Backup found for Mule Runtime distribution: [{}]", muleDistroLocation);
      String backUpDescriptorPrettyFormatted = format("%n%s", formatBackupDescriptor(backupDescriptor.get()));
      LOGGER.debug(backUpDescriptorPrettyFormatted);
    }
    return messages;
  }

  private List<String> checkWritingPermissions(Path muleDistroLocation) {
    List<String> messages = new LinkedList<>();
    try {
      fileSystemService.checkWritingAccess(muleDistroLocation, getAllDefinedPaths());
      LOGGER.debug("Writing permissions found for all expected paths within the Mule distribution [{}]", muleDistroLocation);
    } catch (IOException e) {
      String message =
          String.format("Error validating required writing permissions under Mule distribution [%s].", muleDistroLocation);
      messages.add(message);
      LOGGER.error(message, e);
    }
    return messages;
  }

  private List<String> checkEnoughUsableSpace(Path oldMule, Path newMule) {
    List<String> messages = new LinkedList<>();
    try {
      fileSystemService.checkEnoughUsableSpace(oldMule, newMule, getAllDefinedPaths());
      LOGGER.debug("There's enough disk space for performing the upgrade.");
    } catch (IOException e) {
      String message = "There's not enough disk space for performing the upgrade.";
      messages.add(message);
      LOGGER.error(message);
    }
    return messages;
  }

  private List<String> checkReadingPermissions(Path muleDistroLocation) {
    List<String> messages = new LinkedList<>();
    try {
      fileSystemService.checkReadingAccess(muleDistroLocation, getAllDefinedPaths());
      LOGGER.debug("Reading permissions found for all expected paths within the Mule distribution [{}]", muleDistroLocation);
    } catch (IOException e) {
      String message =
          String.format("Error validating required reading permissions under Mule distribution [%s].", muleDistroLocation);
      messages.add(message);
      LOGGER.error(message, e);
    }
    return messages;
  }

  private List<String> isMuleStopped(Path muleDistroLocation) throws IOException {
    List<String> messages = new LinkedList<>();
    try {
      statusService.checkStopped(muleDistroLocation);
    } catch (IllegalStateException e) {
      String message = "Mule Runtime should be stopped.";
      messages.add(message);
      LOGGER.error(message);
    }
    LOGGER.debug("Mule Distribution located at [{}] is stopped.", muleDistroLocation);
    return messages;
  }
}
