/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.upgrade.tool.service;

import static com.mulesoft.runtime.upgrade.tool.utils.PrettyPrintingFormatter.formatBackupDescriptor;
import static java.lang.String.format;

import com.mulesoft.runtime.upgrade.tool.domain.BackupDescriptor;
import com.mulesoft.runtime.upgrade.tool.domain.MuleDistribution;
import com.mulesoft.runtime.upgrade.tool.service.api.BackupService;
import com.mulesoft.runtime.upgrade.tool.service.api.MuleDistroService;
import com.mulesoft.runtime.upgrade.tool.service.api.StatusPreconditionsValidatorService;
import com.mulesoft.runtime.upgrade.tool.service.api.StatusService;
import com.mulesoft.runtime.upgrade.tool.utils.PrettyPrintingFormatter;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.Path;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * A service for status command.
 */
@Service
public class DefaultStatusService implements StatusService {

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultStatusService.class);

  @Autowired
  private MuleDistroService distroService;

  @Autowired
  private StatusPreconditionsValidatorService statusPreconditionsValidatorService;

  @Autowired
  private BackupService backupService;

  private static final String UPGRADE_MESSAGE =
      "For upgrading: java -jar mule-runtime-upgrade-tool-1.0.0-SNAPSHOT.jar upgrade -o %s -n [\"New Mule Runtime Distribution Path\"]";
  private static final String ROLLBACK_MESSAGE =
      "For rolling back: java -jar mule-runtime-upgrade-tool-1.0.0-SNAPSHOT.jar rollback -o %s";

  private static final int NO_ERRORS_LIST_SIZE = 1;

  /**
   * Runs different processes for showing to the user some relevant information about Mule distribution and possible actions for
   * executing with the tool.
   *
   * @param oldMule Old Mule distro path.
   * @throws FileNotFoundException File not found Exception.
   */
  public void statusCommandExecute(Path oldMule) throws IOException {
    List<String> messagesToPrint = new LinkedList<>();
    List<String> upgradeMessages = preconditionUpgradeStatus(oldMule);
    List<String> rollbackMessages = preconditionRollbackStatus(oldMule);

    boolean upgradePreconditionsMet = upgradeMessages.size() == NO_ERRORS_LIST_SIZE;
    boolean rollbackPreconditionsMet = rollbackMessages.size() == NO_ERRORS_LIST_SIZE;

    String backupPrettyFormat = "";

    String currentMuleDistroVersion = distroService.detectMuleVersion(oldMule);

    messagesToPrint.add("The current Mule Distribution version is " + currentMuleDistroVersion);

    if (distroService.checkClusterModeIsActive(oldMule)) {
      messagesToPrint.add("Mule Runtime Distribution is running in cluster mode.");
    }

    List<String> optionsMessages = new LinkedList<>();

    if (upgradePreconditionsMet || rollbackPreconditionsMet) {
      optionsMessages.add("Possibles commands to execute:");

      if (upgradePreconditionsMet) {
        optionsMessages
            .add(String
                .format("------" + UPGRADE_MESSAGE,
                        oldMule));
      }

      if (rollbackPreconditionsMet) {
        optionsMessages.add(String
            .format("------" + ROLLBACK_MESSAGE, oldMule));

        Optional<BackupDescriptor> backupDescriptor = backupService.readBackupDescriptor(oldMule);

        if (backupDescriptor.isPresent()) {
          backupPrettyFormat = format("%n%s", formatBackupDescriptor(backupDescriptor.get()));
        }
      }
    }

    if (upgradePreconditionsMet) {
      upgradeMessages.add("All the upgrade process preconditions were met.");

      if (distroService.checkMisleadingMuleDirName(oldMule) == true) {
        LOGGER
            .warn("A version in your MULE_HOME directory name was detected: {}, beware it may not match the current installed version inside the directory.",
                  oldMule);
      }

    } else {
      upgradeMessages.add("Not all the upgrade process preconditions were met.");
    }

    if (rollbackPreconditionsMet) {
      rollbackMessages.add("All the rollback process preconditions were met.");
    } else {
      rollbackMessages.add("Not all the rollback process preconditions were met.");
    }

    messagesToPrint.addAll(upgradeMessages);
    messagesToPrint.addAll(rollbackMessages);
    messagesToPrint.addAll(optionsMessages);

    String textToPrint = PrettyPrintingFormatter.formatStatusCommandMessage(messagesToPrint);

    LOGGER.info(String.format("%n%s", textToPrint + backupPrettyFormat));
  }

  public List<String> preconditionUpgradeStatus(Path oldMule) throws IOException {
    LOGGER.info("Checking upgrade preconditions...");
    List<String> upgradeMessages = new LinkedList<>();

    MuleDistribution oldDistro = distroService.initMuleDistro(oldMule);

    upgradeMessages.addAll(statusPreconditionsValidatorService.checkUpgradeStatusOffline(oldDistro));
    if (upgradeMessages.size() == NO_ERRORS_LIST_SIZE) {
      LOGGER.info("All preconditions were met for upgrade process.");
    } else {
      LOGGER.error("Not all preconditions were met for upgrading process.");

    }
    return upgradeMessages;
  }

  public List<String> preconditionRollbackStatus(Path currentMule) throws IOException {
    LOGGER.info("Checking rollback preconditions...");
    List<String> rollbackMessages = new LinkedList<>();
    rollbackMessages.add("Checking rollback process preconditions...");

    rollbackMessages.addAll(statusPreconditionsValidatorService.checkRollbackPreconditions(currentMule));
    if (rollbackMessages.size() == NO_ERRORS_LIST_SIZE) {
      LOGGER.info("All preconditions were met for rolling back process.");
    }

    return rollbackMessages;
  }
}
