/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.upgrade.tool.service.api;

import java.io.IOException;
import java.nio.file.AccessDeniedException;
import java.nio.file.NotDirectoryException;
import java.nio.file.Path;
import java.util.List;

/**
 * A service for file system operations required for the upgrade.
 */
public interface FileSystemService {

  /**
   * Checks whether a given path corresponds to a directory.
   *
   * @param path the path to an expected directory.
   * @throws NotDirectoryException if given path is not a directory.
   */
  void checkIsADirectory(Path path) throws NotDirectoryException;

  /**
   * Check whether the JVM has writing access to all file trees determined by the given relative paths when resolved against the
   * given parent path.
   *
   * @param parentPath    the parent path which is used for resolving the relative paths.
   * @param relativePaths list of all paths whose underlying tree will be validated for writing access.
   * @throws AccessDeniedException if any path does not have writing accessibility.
   * @throws IOException           if any I/O error occurs.
   */
  void checkWritingAccess(Path parentPath, List<Path> relativePaths) throws IOException;

  /**
   * Checks whether the JVM has reading access to all file trees determined by the given relative paths when resolved against the
   * given parent path.
   *
   * @param parentPath    the parent path which is used for resolving the relative paths.
   * @param relativePaths list of all paths whose underlying tree will be validated for reading access.
   * @throws AccessDeniedException if any path does not have reading accessibility.
   * @throws IOException           if any I/O error occurs.
   */
  void checkReadingAccess(Path parentPath, List<Path> relativePaths) throws IOException;

  /**
   * Validates whether there is potentially enough writable space on the file system for the upgrade. The required space is
   * determined either by the space required for the backup creation or the copy of the new files; whoever is the biggest rules.
   * This validation does not guarantee that the process might not fail for lack of space since the conditions on the underlying
   * file system could change in the glimpse of an eye.
   *
   * @param oldMule the directory where the old Mule distro is located.
   * @param newMule the directory where the new Mule distro is located.
   * @param paths   the relative paths of files involved in the upgrade process.
   * @throws {@link IOException} In case of not finding enough usable space.
   */
  void checkEnoughUsableSpace(Path oldMule, Path newMule, List<Path> paths) throws IOException;

  /**
   * Checks whether the underlying OS is Windows.
   *
   * @return true if this is Windows.
   */
  boolean isWindowsOs();

  /**
   * Recursively copy all the files and directories determined by the given relative paths from the given source directory to the
   * given destination directory. If files to be copied already exist the on destination, then they are deleted prior to its copy.
   *
   * @param srcDir        the directory where the files will be copied from.
   * @param destDir       the directory where the files will be copied to.
   * @param paths         the relative paths that will be resolved against the source dir for copying them to the destination
   *                      directory.
   * @param excludedPaths the relative paths that will be resolved against the source dir which are going to be excluded when
   *                      copying them to the destination directory.
   * @param isDryRunMode  if isDryRunMode is true, the method logs in debug level the files and directories that would be copied
   *                      without actually copying them.
   * @throws IOException if any I/O error occurs.
   */
  void copyFiles(Path srcDir, Path destDir, List<Path> paths, List<Path> excludedPaths, boolean isDryRunMode) throws IOException;


  /**
   * Copies a list of files from a Mule Runtime Distribution to another.
   * 
   * @param sourceMuleDistroPath New Mule Runtime Distribution.
   * @param destMuleDistroPath   Old Mule Runtime Distribution.
   * @param fileNamesList        List of files names.
   * @param isDryRunMode         Dry Run Mode flag.
   * @throws IOException IOException.
   */
  void copyFilesInsideADistro(Path sourceMuleDistroPath, Path destMuleDistroPath, List<String> fileNamesList,
                              boolean isDryRunMode)
      throws IOException;


  /**
   * Deletes a list of files from a Mule Runtime Distribution.
   *
   * @param distroPath    Current Mule Runtime Distribution.
   * @param fileNamesList List of files names.
   * @param isDryRunMode  Dry Run Mode flag.
   * @throws IOException IOException.
   */
  void deleteFilesInsideADistro(Path distroPath, List<String> fileNamesList, boolean isDryRunMode) throws IOException;

  /**
   * @param fileA First file to compare the sha256
   * @param fileB Second file to compare the sha256
   * @throws IOException IOException.
   *
   * @return true if the hashes of the files matches and false if it doesn't.
   */
  boolean areHashesBetweenFilesEqual(Path fileA, Path fileB) throws IOException;

  /**
   * @param fileContentA First stream to compare the sha256
   * @param fileContentB Second stream to compare the sha256
   * @throws IOException IOException.
   *
   * @return true if the hashes of the files matches and false if it doesn't.
   */
  boolean areHashesBetweenFilesEqual(byte[] fileContentA, byte[] fileContentB) throws IOException;

  /**
   * Deletes the upgrade-tool folder inside a distro if the folder is empty to try to clean the environment after an upgrade or a
   * backup
   *
   * @param muleHome Mule Home path where it is going to try to delete the empty upgrade-tool folder
   * @param dryRun   Dry Run flag, if is in true don't delete the folder and only do the logging
   * @throws IOException IOException.
   */
  void deleteUpgradeFolderIfEmpty(Path muleHome, boolean dryRun) throws IOException;
}
