/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.upgrade.tool.service.api;

import com.mulesoft.runtime.upgrade.tool.domain.BackupDescriptor;
import com.mulesoft.runtime.upgrade.tool.domain.MuleDistribution;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Optional;

/**
 * A service for backup related operations.
 */
public interface BackupService {

  /**
   * Retrieves the backup creation metadata stored in the YAML file within the backup directory location if any has been
   * previously generated as part of an upgrade process.
   *
   * @param muleDistribution location of the Mule distribution where backup could reside.
   * @return a {@link Optional} of a {@link BackupDescriptor}.
   * @throws IOException In case of any I/O related error while reading the info from the metadata file.
   */
  Optional<BackupDescriptor> readBackupDescriptor(Path muleDistribution) throws IOException;

  /**
   * Creates a backup copy of the given file paths within the Mule distribution directory that will be undergoing the upgrade.
   * Backup creation metadata is stored in a YAML file which is created within the backup directory. The purpose of creating this
   * backup is to be later used in an upgrade rollback operation if needed.
   *
   * @param muleDistribution a {@link MuleDistribution} representing the Mule distro where the backup will be taken from.
   * @param isDryRunMode     if isDryRunMode is true, the method logs in debug level the files and directories that would be
   *                         copied without actually copying them.
   * @throws IOException In case of any I/O related error while creating the backup.
   */
  void create(MuleDistribution muleDistribution, boolean force, boolean isDryRunMode) throws IOException;

  /**
   * Prints a BackUp Descriptor with a pretty format .
   *
   * @param backupDescriptor BackUp Descriptor selected.
   */
  void printBackUpDescriptor(Optional<BackupDescriptor> backupDescriptor);

  /**
   * Checks the integrity of a backup checking the backup descriptor entries match with the current files in the backup folder
   *
   * @param muleDistributionHome location of the Mule distribution where backup could reside.
   *
   * @throws IOException In case of any I/O related error while creating the backup.
   */
  boolean checkIntegrity(Path muleDistributionHome) throws IOException;

  /**
   * Gets the back destination dir where the backup files are going to be placed
   *
   * @param muleDistroHome Path with the mule distribution where is going to be resolved the backup folder path
   * @return Path of the backup folder resolved inside the mule distro passed
   */
  Path getBackupDestDir(Path muleDistroHome);

}
