/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package com.mulesoft.runtime.upgrade.tool.service.api;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.Path;

/**
 * This service manages file descriptor's functionalities.
 */
public interface ConfigFilesService {

  /**
   * This method runs as a precondition of the upgrade command. It gets the config files in the original old and new distribution
   * and compares each config file hash value to detect if any have changes in the new distro.
   *
   * If there are config files that have conflicts the precondition fail and those files are copied from the '${MULE_HOME}/conf'
   * to the "to_review" conflict folder so the user can resolve them. The config files that don't have conflicts are copied from
   * the '${MULE_HOME}/conf' to the "ready" conflict folder.
   *
   * After the user has resolved all the conflicts and moved them to the "ready" folder and runs the upgrade command again the
   * config files present in the "ready" resolved conflict folder are going to be copied to the new distro copied in the method
   * ${@link ConfigFilesService#copyFinalConfigFiles(Path, boolean)}
   *
   * Finally, the new config files are copied too in the "new" aux folder so the user can detect the new changes in an easy way.
   *
   * @param oldMule      The Mule Runtime Distribution path of the old distro.
   * @param newMule      The Mule Runtime Distribution path of the new distro.
   * @param isDryRunMode Flag which in case is in true it avoids copying the config files and deleting the resolved conflicts
   *                     folder
   * @return a boolean with true in case there are config files with conflicts or false otherwise
   */
  boolean checkConflicts(Path oldMule, Path newMule, boolean isDryRunMode) throws IOException;

  /**
   * Copy the config files from the resolved config files folder "ready" to the ${MULE_HOME}/conf during the upgrade command after
   * no conflicts were found
   *
   * @param oldMule The Mule Runtime Distribution path where the resolved config files folder is present to copy the config files
   *                from there during the upgrade.
   * @param dryRun  Flag which in case is in true it avoids copying the config files and deleting the resolved conflicts folder
   */
  void copyFinalConfigFiles(Path oldMule, boolean dryRun) throws IOException;

  /**
   * Gets the folder where the config files to be used after the upgrade. Also in this folder are placed the aux original old and
   * new config files to be used to detect the new changes. This folder is deleted after the upgrade command finishes successfully
   *
   * @return Path with the resolved conflicts folder
   */
  Path getResolvedConflictsFolder();

  /**
   * Gets the documentation page section explaining in detail how to resolve the config files conflicts to be used in the ERROR
   * messages logs.
   *
   * @return String with the complete docs page including the section.
   * @throws FileNotFoundException
   */
  String getSolveConflictsDocsPage();

  /**
   * Gets the corresponding path of the "to_review" folder inside the distribution. In this folder are placed the config files
   * with conflicts
   *
   * @param muleHome Path of the mule distribution inside the 'to_review' path is going to be resolved
   * @return Path with the 'to_review' conflicts folder resolved inside the muleHome
   */
  Path getToReviewFolderInDistro(Path muleHome);

  /**
   * Gets the corresponding path of the "ready" folder inside the distribution. In this folder are placed the config files without
   * conflicts and the ones moved by the user from the "to_review" folder after resolving the conflicts
   *
   * @param muleHome Path of the mule distribution inside the 'ready' path is going to be resolved
   * @return Path with the 'ready' conflicts folder resolved inside the muleHome
   */
  Path getReadyFolderInDistro(Path muleHome);

  /**
   * Extracts the filename of the configFilePath passed as argument and resolves the destination path inside the "to_review"
   * folder .
   *
   * @param muleHome       Path of the mule distribution against the 'to_review' path is going to be resolved
   * @param configFilePath Path of the config file to extract its filename
   * @return Config file path inside the "to_review" folder
   */
  Path getToReviewConfigFileDestination(Path muleHome, String configFilePath);

  /**
   * Extracts the filename of the configFilePath passed as argument and resolves the destination path inside the "ready" folder.
   *
   * @param muleHome       Path of the mule distribution against the 'ready' path is going to be resolved
   * @param configFilePath Path of the config file to extract its filename
   * @return Config file path inside the "ready" folder
   */
  Path getReadyConfigFileDestination(Path muleHome, String configFilePath);

}
