/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.upgrade.tool.service;

import static java.lang.String.format;
import static java.nio.charset.StandardCharsets.UTF_8;

import com.mulesoft.runtime.upgrade.tool.service.api.FileSystemService;
import com.mulesoft.runtime.upgrade.tool.service.api.MuleStatusService;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;

import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * A service for Mule Runtime process-related operations.
 */
@Service
public class DefaultMuleStatusService implements MuleStatusService {

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultMuleStatusService.class);

  private static final Path UNIX_MULE_BINARY = Paths.get("bin", "mule");
  private static final Path WINDOWS_MULE_BINARY = Paths.get("bin", "mule.bat");

  private static final String NOT_RUNNING_EXPECTED_UNIX_OUTPUT_PATTERN = "(?s).*not running.*";
  private static final String NOT_RUNNING_EXPECTED_WINDOWS_OUTPUT_PATTERN = "(?s).*(Running: No|not installed).*";

  @Autowired
  private FileSystemService fileSystemService;

  /**
   * Checks whether a Mule Runtime distribution is stopped.
   *
   * @throws IllegalStateException if Mule runtime is running.
   * @throws IOException           if there's any issue while interacting with the mule binary.
   */
  @Override
  public void checkStopped(Path muleLocation) throws IOException {
    LOGGER.debug("Checking if Mule runtime at [{}] is stopped", muleLocation);

    String[] queryMuleStatusCommand = getQueryMuleStatusCommand(muleLocation);
    String processOutput = sanitizeProcessOutput(executeQueryStatusCommand(queryMuleStatusCommand));
    if (!processOutputMatchesExpectedStoppedPattern(processOutput)) {
      throw new IllegalStateException(format("Mule runtime at [%s] is running", muleLocation));
    }
  }

  private String[] getQueryMuleStatusCommand(Path muleLocation) throws FileNotFoundException {
    String[] queryMuleStatusCommand;
    String muleBinary = resolveMuleBinary(muleLocation).toAbsolutePath().toString();
    queryMuleStatusCommand = new String[] {muleBinary, "status"};
    return queryMuleStatusCommand;
  }

  protected Path getMuleBinaryPath() {
    return fileSystemService.isWindowsOs() ? WINDOWS_MULE_BINARY : UNIX_MULE_BINARY;
  }

  protected String getEnvVariable(String envVariableName) {
    return System.getenv(envVariableName);
  }

  protected String executeQueryStatusCommand(String[] command) throws IOException {
    String stringCommand = Arrays.toString(command);
    LOGGER.debug("Figuring out if Mule is running by using command {}", stringCommand);
    Process status = new ProcessBuilder(command).redirectErrorStream(true).start();
    status.getOutputStream().close();
    return IOUtils.toString(status.getInputStream(), UTF_8);
  }

  private Path resolveMuleBinary(Path muleLocation) throws FileNotFoundException {
    Path muleBinary = muleLocation.resolve(getMuleBinaryPath());
    if (!muleBinary.toFile().exists()) {
      throw new FileNotFoundException(format("Mule binary does not exists at the expected location [%s]", muleBinary));
    }
    return muleBinary;
  }

  private String sanitizeProcessOutput(String processOutput) {
    return processOutput.replace("\0", "");
  }

  private boolean processOutputMatchesExpectedStoppedPattern(String processOutput) {
    return fileSystemService.isWindowsOs() ? processOutput.matches(NOT_RUNNING_EXPECTED_WINDOWS_OUTPUT_PATTERN)
        : processOutput.matches(NOT_RUNNING_EXPECTED_UNIX_OUTPUT_PATTERN);
  }
}
