/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package com.mulesoft.runtime.upgrade.tool.service;

import static com.mulesoft.runtime.upgrade.tool.utils.JarFileUtils.getFileContentInByteArray;
import static com.mulesoft.runtime.upgrade.tool.utils.JarFileUtils.getJarEntries;
import static org.apache.commons.io.FileUtils.copyInputStreamToFile;
import static org.apache.commons.io.FileUtils.deleteDirectory;
import static org.apache.commons.io.FileUtils.deleteQuietly;
import static org.apache.commons.io.FileUtils.writeByteArrayToFile;

import com.mulesoft.runtime.upgrade.tool.service.api.ConfigFilesService;
import com.mulesoft.runtime.upgrade.tool.service.api.FileSystemService;
import com.mulesoft.runtime.upgrade.tool.service.api.MuleDistroService;
import com.mulesoft.runtime.upgrade.tool.service.api.UpgradeConfigService;
import com.mulesoft.runtime.upgrade.tool.utils.ClassLoaderService;
import com.mulesoft.runtime.upgrade.tool.utils.JarFileUtils;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Service;

/**
 * This service manages file descriptor's functionalities.
 */
@Service
public class DefaultConfigFilesService implements ConfigFilesService {

  @Autowired
  private MuleDistroService muleDistroService;

  @Autowired
  private ClassLoaderService classLoaderService;

  @Autowired
  private FileSystemService fileSystemService;

  @Autowired
  private UpgradeConfigService upgradeConfigService;

  public static final String TO_REVIEW_FOLDER_NAME = "to_review";
  public static final String READY_FOLDER_NAME = "ready";
  public static final String ORIGINAL_FILES_NEW_DISTRO_FOLDER = "new";
  public static final String LIC_MULE_FILE = ".lic-mule";

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultConfigFilesService.class);
  static final String CONFIGS_FILES_JAR_NAME_PREFIX = "mule-config-files-";
  // We need to hardcode the '/' because resolving the file separator of the OS (in windows '\') fails to find the resource in the
  // upgrade tool
  static final String CONFIGS_FILES_JARS_DIRECTORY_INSIDE_UPGRADE_TOOL = "mule-config-files/";

  @Override
  public Path getResolvedConflictsFolder() {
    return Paths.get(upgradeConfigService.getUpgradeToolFolderName(),
                     upgradeConfigService.getConfigFilesResolvedConflictsFolderName());
  }

  public String getSolveConflictsDocsPage() {
    return upgradeConfigService.getDocsPage() + "#" + upgradeConfigService.getDocsSolveConfigConflictsSection();
  }

  String getConfigsFileJarPath(Path muleHome) throws IOException {
    if (checkConfigsFileJarExistenceInsideUpgradeTool(muleHome)) {
      String version = muleDistroService.detectMuleVersion(muleHome);
      return extractConfigFilesJarFromUpgradeTool(version, muleHome);
    } else if (checkConfigsFileJarExistenceInsideMuleDistro(muleHome)) {
      return resolveConfigsFilesJarPathInDistro(muleHome);
    } else {
      throw new FileNotFoundException("The Mule Runtime config Jar file was NOT found neither in the Mule Distribution "
          + muleHome
          + " nor inside the upgrade tool configs files jars.");
    }
  }

  String extractConfigFilesJarFromUpgradeTool(String version, Path muleHome) throws IOException {
    Optional<String> configFilesJarURLOptional = resolveConfigsFilesJarPathInUpgradeTool(version);

    if (configFilesJarURLOptional.isPresent()) {
      String configFilesJarInUpgradeToolPath = configFilesJarURLOptional.get();
      File destExtractedFile = getJarInUpgradeToolExtractionDestination(muleHome, configFilesJarInUpgradeToolPath).toFile();
      Resource configsFileJarInsideTool =
          classLoaderService.getResourceFromClasspath(configFilesJarInUpgradeToolPath);
      copyInputStreamToFile(configsFileJarInsideTool.getInputStream(), destExtractedFile);
      return destExtractedFile.getAbsolutePath();
    } else {
      throw new FileNotFoundException("The Configs Jar file was not found inside the upgrade tool configs files jars.");
    }
  }

  Path getJarInUpgradeToolExtractionDestination(Path muleHome, String configFilesJarInUpgradeToolPath) {
    String configFilesJarName = Paths.get(configFilesJarInUpgradeToolPath).getFileName().toString();
    return getResolvedConflictsFolderInDistro(muleHome).resolve(configFilesJarName);
  }

  boolean checkConfigsFileJarExistenceInsideUpgradeTool(Path muleHome) throws FileNotFoundException {
    String currentVersion = muleDistroService.detectMuleVersion(muleHome);
    return checkConfigsFileJarExistenceInsideUpgradeTool(currentVersion);
  }

  boolean configFileIsInReadyFolder(Path muleHome, String configFilePath) {
    Path configFileResolvedDestination = getReadyConfigFileDestination(muleHome, configFilePath);
    return Files.exists(configFileResolvedDestination);
  }

  public Path getResolvedConflictsFolderInDistro(Path muleHome) {
    return muleHome.resolve(getResolvedConflictsFolder());
  }

  boolean checkConfigsFileJarExistenceInsideUpgradeTool(String version) {
    Optional<String> configsFileJarInUpgradeTool = resolveConfigsFilesJarPathInUpgradeTool(version);

    return configsFileJarInUpgradeTool.isPresent();
  }

  boolean checkConfigsFileJarExistenceInsideMuleDistro(Path muleHome) throws FileNotFoundException {
    String configsFileJarInDistro = resolveConfigsFilesJarPathInDistro(muleHome);

    if (Files.exists(Paths.get(configsFileJarInDistro))) {
      LOGGER.debug("Mule Runtime config files jar [{}] was found inside the distribution.", configsFileJarInDistro);
      return true;
    } else {
      LOGGER.debug("Mule Runtime config files jar [{}] was NOT found inside the distribution.", configsFileJarInDistro);
      return false;
    }
  }

  String resolveConfigsFilesJarName(String version) {
    return CONFIGS_FILES_JAR_NAME_PREFIX + version + ".jar";
  }

  String resolveConfigsFilesJarNameInUpgradeTool(String version) {
    return CONFIGS_FILES_JARS_DIRECTORY_INSIDE_UPGRADE_TOOL + resolveConfigsFilesJarName(version);
  }

  Optional<String> resolveConfigsFilesJarPathInUpgradeTool(String version) {
    String configFilesJarPathInUpgradeTool = resolveConfigsFilesJarNameInUpgradeTool(version);
    URL configsFileJarInsideUpgradeToolURL =
        classLoaderService.getURLByResourcePath(configFilesJarPathInUpgradeTool);
    if (configsFileJarInsideUpgradeToolURL == null) {
      LOGGER.debug("Mule Runtime config files jar was NOT found in upgrade tool under resources path [{}]",
                   configFilesJarPathInUpgradeTool);
      return Optional.empty();
    }
    LOGGER.debug("Mule Runtime config files jar was found in upgrade tool under resources path [{}]",
                 configFilesJarPathInUpgradeTool);
    return Optional.of(configFilesJarPathInUpgradeTool);
  }

  String resolveConfigsFilesJarPathInDistro(Path muleHome) throws FileNotFoundException {
    String currentVersion = muleDistroService.detectMuleVersion(muleHome);
    Path configsFileJarWithVersion =
        muleDistroService.getLibMuleJarDirectoryPath().resolve(resolveConfigsFilesJarName(currentVersion));
    return muleHome.resolve(configsFileJarWithVersion).toAbsolutePath().toString();
  }

  @Override
  public boolean checkConflicts(Path oldMule, Path newMule, boolean isDryRunMode) throws IOException {
    LOGGER.debug("Checking Mule Runtime Config files conflicts between distributions [{}] and [{}]...", oldMule, newMule);
    validateConfigFilesJarsExistencesInBothDistros(oldMule, newMule);

    String configsFileJarPathInOldDistro;
    String configsFileJarPathInNewDistro;
    try {
      configsFileJarPathInOldDistro = getConfigsFileJarPath(oldMule);
      configsFileJarPathInNewDistro = getConfigsFileJarPath(newMule);
    } catch (FileNotFoundException e) {
      throw new IOException("Upgrading from/to this version caused an unexpected error trying to obtain the Mule Runtime configs files. Please create a ticket to support.",
                            e);
    }

    List<String> configFilesInOldDistroJar = getJarEntries(configsFileJarPathInOldDistro);

    boolean configFilesCheckIsFine = true;
    for (String configFilePathInOldDistroJar : configFilesInOldDistroJar) {
      byte[] configFileContentInOldDistro =
          getFileContentInByteArray(configsFileJarPathInOldDistro, configFilePathInOldDistroJar);

      byte[] configFileContentInNewDistro = new byte[0];
      boolean configFileIsPresentInNewDistro =
          JarFileUtils.isFilePresent(configsFileJarPathInNewDistro, configFilePathInOldDistroJar);
      if (configFileIsPresentInNewDistro) {
        configFileContentInNewDistro = getFileContentInByteArray(configsFileJarPathInNewDistro, configFilePathInOldDistroJar);
      }

      if (configFileIsInReadyFolder(oldMule, configFilePathInOldDistroJar)) {
        LOGGER
            .debug("The config File '{}' was found in the '{}' folder. Will proceed without checking conflicts in this config file.",
                   configFilePathInOldDistroJar, getReadyFolderInDistro(oldMule));
      } else if (configFileWasRemovedFromDistro(oldMule, configFilePathInOldDistroJar)) {
        LOGGER
            .debug(
                   "The config File '{}' was removed from the original version. Make sure you don't need it to keep your system working.",
                   configFilePathInOldDistroJar);
      } else if (!configFileIsPresentInNewDistro) {
        LOGGER
            .debug(
                   "The config File '{}' was NOT found in the new distro config files '[{}]' jar. Will proceed copying the user config file to the ready folder.",
                   configFilePathInOldDistroJar, configsFileJarPathInNewDistro);
        copyUserConfigFileWoutConflictsToReadyFolder(oldMule, configFilePathInOldDistroJar, isDryRunMode);
      } else if (!userConfigFileHasChanges(oldMule, configFilePathInOldDistroJar, configFileContentInOldDistro)) {
        copyNewConfigFileToReadyFolder(oldMule, configsFileJarPathInNewDistro, configFilePathInOldDistroJar,
                                       configFileContentInNewDistro, isDryRunMode);
      } else if (configFileHasChanges(configFileContentInOldDistro, configFileContentInNewDistro)) {
        configFilesCheckIsFine = false;
        showConfigFileWithConflictErrorMessage(oldMule, configFilePathInOldDistroJar);

        copyUserConfigFileWithConflictsToReviewFolder(oldMule, configFilePathInOldDistroJar, isDryRunMode);
        copyOriginalConfigFileInNewConflictsFolderAsReference(oldMule, newMule, configFilePathInOldDistroJar,
                                                              configsFileJarPathInNewDistro,
                                                              configFileContentInNewDistro, isDryRunMode);
      } else {
        copyUserConfigFileWoutConflictsToReadyFolder(oldMule, configFilePathInOldDistroJar, isDryRunMode);
      }
    }

    deleteAuxConfigFileJars(oldMule, configsFileJarPathInNewDistro, configsFileJarPathInNewDistro);

    return configFilesCheckIsFine;
  }

  private void showConfigFileWithConflictErrorMessage(Path oldMule, String configFilePathInOldDistroJar) {
    String errorMessage =
        "The config file '{}' was updated in the new version, make sure the one you want to use with the changes needed is in the '{}' folder [{}] as it is required before being able to re-run the upgrade command again.";
    LOGGER.error(errorMessage, configFilePathInOldDistroJar, getReadyFolderName(),
                 getReadyConfigFileDestination(oldMule, configFilePathInOldDistroJar));

  }

  private boolean configFileWasRemovedFromDistro(Path oldMule, String configFileNameInOldDistroJar) {
    return !Files.isRegularFile(Paths.get(oldMule.toString(), configFileNameInOldDistroJar));
  }

  private void validateConfigFilesJarsExistencesInBothDistros(Path oldMule, Path newMule) throws FileNotFoundException {
    if (checkConfigFilesJarsExistencesInsideBothDistributions(oldMule, newMule)) {
      LOGGER.debug("Mule Runtime config files jars successfully found for both distributions [{}] and [{}]", oldMule, newMule);
    } else {
      throw new FileNotFoundException((String
          .format("Mule Runtime config files jars NOT found for both distributions [%s] and [%s]", oldMule, newMule)));
    }
  }

  void deleteAuxConfigFileJars(Path oldMuleHome, String configsFileJarPathInOldDistro, String configsFileJarPathInNewDistro) {
    Path configsFileJarOldDistro = Paths.get(configsFileJarPathInOldDistro);
    Path configsFileJarNewDistro = Paths.get(configsFileJarPathInNewDistro);

    deleteIfExistsInsideResolvedConflictsFolder(configsFileJarOldDistro, oldMuleHome);
    deleteIfExistsInsideResolvedConflictsFolder(configsFileJarNewDistro, oldMuleHome);
  }

  private void deleteIfExistsInsideResolvedConflictsFolder(Path configsFileJar, Path muleHome) {
    Path resolvedConflictsFolder = getResolvedConflictsFolderInDistro(muleHome);

    if (Files.exists(configsFileJar) && configsFileJar.getParent().equals(resolvedConflictsFolder)) {
      deleteQuietly(configsFileJar.toFile());
    }
  }

  void copyUserConfigFileWoutConflictsToReadyFolder(Path muleHome, String configFileName, boolean isDryRun) throws IOException {
    File sourceFile = getConfConfigFileDestination(muleHome, configFileName).toFile();
    File destFile = getReadyConfigFileDestination(muleHome, configFileName).toFile();

    fileSystemService.copyFile(sourceFile, destFile, isDryRun);
  }

  public String getToReviewFolderName() {
    return TO_REVIEW_FOLDER_NAME;
  }

  public String getReadyFolderName() {
    return READY_FOLDER_NAME;
  }

  String getNewFolderName() {
    return ORIGINAL_FILES_NEW_DISTRO_FOLDER;
  }

  public Path getToReviewFolderInDistro(Path muleHome) {
    return getResolvedConflictsFolderInDistro(muleHome).resolve(getToReviewFolderName());
  }

  public Path getToReviewConfigFileDestination(Path muleHome, String configFilePath) {
    Path configFileName = Paths.get(configFilePath).getFileName();
    return getToReviewFolderInDistro(muleHome).resolve(configFileName);
  }

  public Path getReadyFolderInDistro(Path muleHome) {
    return getResolvedConflictsFolderInDistro(muleHome).resolve(getReadyFolderName());
  }

  public Path getReadyConfigFileDestination(Path muleHome, String configFilePath) {
    Path configFileName = Paths.get(configFilePath).getFileName();
    return getReadyFolderInDistro(muleHome).resolve(configFileName);
  }

  Path getNewConfigFileDestination(Path muleHomeOld, Path muleHomeNew, String configFilePath) {
    Path configFileName = Paths.get(configFilePath).getFileName();
    return getNewFolderInDistro(muleHomeOld, muleHomeNew).resolve(configFileName);
  }

  public Path getConfFolderInDistro(Path muleHome) {
    return muleHome.resolve(muleDistroService.getConfigFolder());
  }

  public Path getConfConfigFileDestination(Path muleHome, String configFilePath) {
    Path configFileName = Paths.get(configFilePath).getFileName();
    return getConfFolderInDistro(muleHome).resolve(configFileName);
  }

  void copyUserConfigFileWithConflictsToReviewFolder(Path muleHome, String configFilePath, boolean isDryRun) throws IOException {
    File sourceFile = getConfConfigFileDestination(muleHome, configFilePath).toFile();
    File destFile = getToReviewConfigFileDestination(muleHome, configFilePath).toFile();

    fileSystemService.copyFile(sourceFile, destFile, isDryRun);
  }

  private void copyOriginalConfigFileInNewConflictsFolderAsReference(Path muleHomeOld, Path muleHomeNew,
                                                                     String configFilePath, String configsFileJarPathInNewDistro,
                                                                     byte[] newConfigFileContent, boolean isDryRun)
      throws IOException {
    File destFileNewInDistro = getNewConfigFileDestination(muleHomeOld, muleHomeNew, configFilePath).toFile();
    LOGGER.trace("Copying file [{}{}] to [{}]...", configsFileJarPathInNewDistro, configFilePath, destFileNewInDistro);
    if (!isDryRun) {
      writeByteArrayToFile(destFileNewInDistro, newConfigFileContent);
    }
  }

  Path getNewFolderInDistro(Path muleHomeOld, Path muleHomeNew) {
    return getResolvedConflictsFolderInDistro(muleHomeOld).resolve(
                                                                   ORIGINAL_FILES_NEW_DISTRO_FOLDER
                                                                       + "-"
                                                                       + muleHomeNew
                                                                           .getFileName());
  }

  boolean configFileHasChanges(byte[] configFileContentInOldDistro, byte[] configFileContentInNewDistro)
      throws IOException {
    return !fileSystemService.areHashesBetweenFilesEqual(configFileContentInOldDistro, configFileContentInNewDistro);
  }

  boolean checkConfigFilesJarsExistencesInsideBothDistributions(Path oldMulePath, Path newMulePath)
      throws FileNotFoundException {
    LOGGER.debug("Checking mule-config-files jar existence...");
    return checkConfigsFileJarExistenceInsideMuleDistroOrInsideUpgradeTool(oldMulePath)
        && checkConfigsFileJarExistenceInsideMuleDistroOrInsideUpgradeTool(newMulePath);
  }

  boolean checkConfigsFileJarExistenceInsideMuleDistroOrInsideUpgradeTool(Path muleHome)
      throws FileNotFoundException {
    return checkConfigsFileJarExistenceInsideMuleDistro(muleHome) || checkConfigsFileJarExistenceInsideUpgradeTool(muleHome);
  }

  @Override
  public void copyFinalConfigFiles(Path oldMule, boolean isDryRunMode) throws IOException {
    Path readyFolderInDistro = getReadyFolderInDistro(oldMule);
    Path confFolderInDistro = oldMule.resolve(muleDistroService.getConfigFolder());

    LOGGER.debug("Copying mule runtime distribution config files from [{}] to [{}].",
                 readyFolderInDistro,
                 oldMule.resolve(muleDistroService.getConfigFolder()));

    List<Path> includedPaths = new ArrayList<>();
    List<Path> excludedPaths = new ArrayList<>();

    fileSystemService.copyFiles(readyFolderInDistro, confFolderInDistro, includedPaths, excludedPaths, isDryRunMode);

    if (!isDryRunMode) {
      Path resolvedConflictsFolder = getResolvedConflictsFolderInDistro(oldMule);
      deleteDirectory(resolvedConflictsFolder.toFile());
    }
  }

  private void copyNewConfigFileToReadyFolder(Path muleHomeOld, String configsFileJarPathInNewDistro, String configFilePath,
                                              byte[] newConfigFileContent, boolean isDryRun)
      throws IOException {
    File destFileInReadyFolder = getReadyConfigFileDestination(muleHomeOld, configFilePath).toFile();

    LOGGER.trace("Copying file [{}{}] to [{}]...", configsFileJarPathInNewDistro, configFilePath, destFileInReadyFolder);
    if (!isDryRun) {
      writeByteArrayToFile(destFileInReadyFolder, newConfigFileContent);
    }
  }

  private boolean userConfigFileHasChanges(Path muleHome, String configFilePath, byte[] originalOldDistroConfigFileContent)
      throws IOException {
    byte[] userConfigFileContent = FileUtils.readFileToByteArray(getConfConfigFileDestination(muleHome, configFilePath).toFile());
    return configFileHasChanges(userConfigFileContent, originalOldDistroConfigFileContent);
  }
}
