/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.upgrade.tool.service;

import static java.lang.String.format;

import com.mulesoft.runtime.upgrade.tool.service.api.DistroPatchService;

import java.io.File;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.commons.io.filefilter.RegexFileFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

/**
 * A service for Mule distribution patches handling.
 */
@Service
public class DefaultDistroPatchService implements DistroPatchService {

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultDistroPatchService.class);

  private static final String PATCHES_NAME_REGEX = "^.*\\.jar$";

  protected static final Path PATCHES_DIR = Paths.get("lib", "patches");

  /**
   * Retrieves all the file paths of the applied patches of a given Mule Runtime Distribution if any.
   *
   * @param muleDistribution location of the Mule distribution where patches could reside.
   * @throws FileNotFoundException if patches dir is not found within the given mule distro location.
   */
  @Override
  public List<File> getAppliedPatches(Path muleDistribution) throws FileNotFoundException {
    Path patchesDir = muleDistribution.resolve(PATCHES_DIR);
    List<File> patches = findPatchesInDistro(muleDistribution);

    if (!patches.isEmpty()) {
      LOGGER.debug("The following patches were found at {}: {}", patchesDir, patches);
    } else {
      LOGGER.debug("No applied patches found at {}", patchesDir);
    }

    return patches;
  }

  /**
   * Retrieves all the applied patches of a given Mule Runtime Distribution if any using relative paths.
   *
   * @param muleDistribution location of the Mule distribution where patches could reside.
   * @throws FileNotFoundException if patches dir is not found within the given mule distro location.
   */
  @Override
  public List<File> getAppliedPatchesWithRelativePaths(Path muleDistribution) throws FileNotFoundException {
    List<File> appliedPatches = findPatchesInDistro(muleDistribution);
    return appliedPatches.stream().map(appliedPatch -> muleDistribution.relativize(appliedPatch.toPath()).toFile())
        .collect(Collectors.toList());
  }

  /**
   * Get patches dir inside the Mule Distribution path passed as parameter
   *
   * @param muleDistribution location of the Mule distribution to get the patches dir.
   * @return The PATCHES_DIR resolved path from the muleDistribution parameter passed
   */
  @Override
  public Path getPatchesDir(Path muleDistribution) {
    return muleDistribution.resolve(PATCHES_DIR);
  }


  /**
   * this function searches for patches in a specific directory.
   *
   * @param patchesDir location where is patches' directory.
   * @return List of files or null.
   * @throws FileNotFoundException
   */
  private List<File> findPatches(File patchesDir) throws FileNotFoundException {
    if (!patchesDir.isDirectory()) {
      throw new FileNotFoundException(format("Expected patches dir does not exists at: %s", patchesDir));
    }
    FileFilter fileFilter = new RegexFileFilter(PATCHES_NAME_REGEX);
    return Arrays.asList(patchesDir.listFiles(fileFilter));
  }

  private List<File> findPatchesInDistro(Path muleDistribution) throws FileNotFoundException {
    Path patchesDir = muleDistribution.resolve(PATCHES_DIR);
    return findPatches(patchesDir.toFile());
  }
}
