/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.runtime.upgrade.tool.service;

import com.mulesoft.runtime.upgrade.tool.service.api.UserInteractionService;

import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.util.Scanner;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

/**
 * Service for interacting with the end user.
 */
@Service
public class DefaultUserInteractionService implements UserInteractionService {

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultUserInteractionService.class);

  private static final String YES_NO_REGEX = "(?i)^(y(es)*|n(o)*)$";

  protected static final String YES_NO_PROMPT = "To confirm please, enter [yes, no]";

  protected static final String DEFAULT_NO_SUFFIX_PROMPT = " (default: no): ";

  protected static final String DEFAULT_YES_SUFFIX_PROMPT = " (default: yes): ";

  private InputStream in;

  private PrintStream out;

  public DefaultUserInteractionService() {
    in = System.in;
    out = System.out;
  }

  /**
   * Requests yes/no confirmation for an action given its description. Interaction takes place by Java standard input and output
   * streams. Take into account this will imply blocking calls.
   *
   * @param desc                a short description of the action to confirm for logging purposes.
   * @param messageToUser       the message that should be shown to the user for requesting confirmation.
   * @param defaultConfirmation false, if default confirmation answer should be No, true otherwise.
   *
   * @return true if the user accepts, false otherwise.
   * @throws IOException on any I/O related exception like no Console associated to the JVM.
   */
  @Override
  public boolean confirmAction(String desc, String messageToUser, boolean defaultConfirmation) throws IOException {
    LOGGER.debug("Requesting confirmation for action {}", desc);
    checkConsoleExistence();

    Scanner scanner = new Scanner(in);
    out.println(messageToUser);
    String prompt = YES_NO_PROMPT + (defaultConfirmation ? DEFAULT_YES_SUFFIX_PROMPT : DEFAULT_NO_SUFFIX_PROMPT);

    String input = "";
    boolean validInput = false;
    while (!validInput) {
      out.printf(prompt);
      input = scanner.nextLine().trim();
      validInput = input.matches(YES_NO_REGEX) || input.isEmpty();
    }

    boolean confirmed = (input.isEmpty() && defaultConfirmation) || input.toLowerCase().startsWith("y");
    LOGGER.debug("Confirmation for action '{}', was: {}", desc, confirmed);

    return confirmed;
  }

  public void setIn(InputStream in) {
    this.in = in;
  }

  public void setOut(PrintStream out) {
    this.out = out;
  }

  protected void checkConsoleExistence() throws IOException {
    if (System.console() == null) {
      throw new IOException("No Console associated with the current Java virtual machine for interactive user input");
    }
  }

}
