/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package com.mulesoft.runtime.upgrade.tool.service.api;

import com.mulesoft.runtime.upgrade.tool.domain.AssemblyEntry;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.Path;
import java.util.List;


/**
 * This service manages file descriptor's functionalities.
 */
public interface DescriptorService {

  /**
   * This method creates file descriptor of the current client mule and then compares with the original file descriptor of the
   * same version.
   *
   * @param oldMulePath Path of the current mule distro.
   * @throws IOException I/O Exception.
   */
  void compareMuleDistros(Path oldMulePath) throws IOException;

  /**
   * This method recursively gets the assembly entries from the files inside a directory
   *
   * @param directory Path of the directory to get the assembly entries list.
   * @throws IOException I/O Exception.
   */
  List<AssemblyEntry> getEntriesInPath(Path directory) throws IOException;

  /**
   * This method checks the integrity of the files inside a directory and a list of assembly entries
   *
   * @param entries      List of assembly entries
   * @param dirWithFiles Path of the directory to be checked against the assembly entries list
   * @throws IOException I/O Exception.
   */
  boolean checkIntegrity(List<AssemblyEntry> entries, Path dirWithFiles) throws IOException;


  /**
   * Looking for File Descriptor inside the tool, and if doesn't exist there, try to find it inside the Mule Runtime Distribution.
   * 
   * @param currentVersion Mule Runtime Distribution version.
   * @param mulePath       Mule Runtime Distribution Path.
   * @return Assembly Entries List.
   */
  List<AssemblyEntry> getAssemblyEntryListByFileDescriptor(String currentVersion, Path mulePath) throws IOException;

  /**
   * Checks File Descriptor existence inside a specific mule distro and inside the tool.
   *
   * @param mulePath Mule Runtime Distribution path.
   * @return True if exists, false if doesn't exist.
   * @throws FileNotFoundException
   */
  boolean checkFileDescriptorExistenceInsideMuleDistroOrInsideUpgradeTool(Path mulePath) throws IOException;

  /**
   * Makes a list with the names of all files which appears in the File Descriptor of an specific Mule Runtime Distribution.
   * 
   * @param muleDistroPath Mule Runtime Distribution Path.
   * @return List of file names.
   * @throws IOException
   */
  List<String> getFilesNamesList(Path muleDistroPath) throws IOException;
}
