/*
 * Copyright (c) 2009 Mysema Ltd.
 * All rights reserved.
 * 
 */
package com.mysema.commons.l10n.support;

import java.util.Iterator;
import java.util.Locale;
import java.util.NoSuchElementException;

/**
 * LocaleIterator provides iteration over locales
 * 
 * @author Samppa Saarela
 * @version $Id: LocaleIterator.java 2386 2007-02-24 18:04:04Z tiwe $
 */
public final class LocaleIterator implements Iterator<Locale> {
    /*
     * <ol> <li>baseName + "_" + language1 + "_" + country1 + "_" +
     * variant1</li> <li>baseName + "_" + language1 + "_" + country1 </li>
     * <li>baseName + "_" + language1 </li> <li>baseName + "_" + language2 + "_"
     * + country2 + "_" + variant2</li> <li>baseName + "_" + language2 + "_" +
     * country2</li> <li>baseName + "_" + language2 </li> <li>baseName</li>
     * </ol>
     */
    private static final int END = -1;

    private static final int B = 0;

    private static final int BL = 1;

    private static final int B_C = 2;

    private static final int BLC = 3;

    // private static final int B__V = 4;
    private static final int BL_V = 5;

    private static final int B_CV = 6;

    private static final int BLCV = 7;

    private Locale next;

    private boolean incDefault;

    private int stage;

    /**
     * @param locale
     * @param incDefault
     */
    public LocaleIterator(Locale locale, boolean incDefault) {
        next = locale;
        this.incDefault = incDefault;
        stage = getLocaleStage(next);
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.util.Iterator#hasNext()
     */
    public boolean hasNext() {
        return next != null;
    }

    /**
     * @param locale
     * @return
     */
    public int getLocaleStage(Locale locale) {
        int[] clen = new int[] { locale.getLanguage().length(),
                locale.getCountry().length(), locale.getVariant().length() };
        int localStage = 0;
        for (int i = 0, b = 1; i < 3; i++, b *= 2) {
            if (clen[i] > 0) {
                localStage |= b;
            }
        }
        return localStage;
    }

    /**
     * @param locale
     * @return
     */
    private Locale getNext(Locale locale) {
        switch (stage) {
        case B:
            stage = END;
            return null;
        case BL:
        case B_C:
            stage = B;
            return LocaleUtil.BASE_LOCALE;
        case BLC:
        case BL_V:
            stage = BL;
            return new Locale(locale.getLanguage());
        case B_CV:
            stage = B_C;
            return new Locale("", locale.getCountry());
        case BLCV:
            stage = BLC;
            return new Locale(locale.getLanguage(), locale.getCountry());
        default:
            throw new IllegalStateException("Illegal locale stage: " + stage);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.util.Iterator#next()
     */
    public Locale next() {
        if (next == null) {
            throw new NoSuchElementException();
        }
        if (incDefault) {
            if (stage == B) {
                next = Locale.getDefault();
                incDefault = false;
                stage = getLocaleStage(next);
            } else if (Locale.getDefault().equals(next)) {
                incDefault = false;
            }
        }
        Locale rv = next;
        next = getNext(next);
        return rv;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.util.Iterator#remove()
     */
    public void remove() {
        throw new UnsupportedOperationException();
    }

    // public static void main(String[] args) {
    // boolean incDefault = Boolean.parseBoolean(args[0]);
    // for (int i = 1; i < args.length; i++) {
    // System.out.println("::" + args[i] + "::");
    // for (Iterator<Locale> iter = new LocaleIterator(LocaleUtil
    // .parseLocale(args[i]), incDefault); iter.hasNext();) {
    // System.out.println(">" + iter.next() + "<");
    // }
    // }
    // }
}
