/*
 * Copyright (c) 2009 Mysema Ltd.
 * All rights reserved.
 * 
 */
package com.mysema.commons.l10n.support;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

/**
 * LocaleUtil provides various Locale related methods
 * 
 * @author Samppa Saarela
 * @version $Id: LocaleUtil.java 2773 2007-09-21 16:15:36Z tiwe $
 */
public final class LocaleUtil {

    private static final int BASE = 0;

    private static final int LANG = 1;

    private static final int LANG_COUNTRY = 2;

    private static final int LANG_COUNTRY_VARIANT = 3;

    public static final Locale BASE_LOCALE = new Locale("");

    private LocaleUtil() {
    }

    /**
     * @param locale
     * @return
     */
    public static Locale parseLocale(String locale) {
        if (locale == null) {
            return BASE_LOCALE;
        }
        String[] s = locale.split("[_-]", LANG_COUNTRY_VARIANT);
        switch (s.length) {
        case BASE:
            return BASE_LOCALE;
        case LANG:
            return new Locale(s[0]);
        case LANG_COUNTRY:
            return new Locale(s[0], s[1]);
        case LANG_COUNTRY_VARIANT:
            return new Locale(s[0], s[1], s[2]);
        default:
            throw new IllegalArgumentException(locale);
        }
    }

    public static String toLang(Locale locale) {
        StringBuilder sb = new StringBuilder(5);
        if (isNotEmpty(locale.getLanguage())) {
            sb.append(locale.getLanguage().toLowerCase());
        }
        if (isNotEmpty(locale.getCountry())) {
            sb.append('-');
            sb.append(locale.getCountry().toLowerCase());
        }
        return sb.toString();
    }

    public static <V> V getLocalized(Map<Locale, V> values,
            Iterable<Locale> locales, V defaultValue) {
        return getLocalized(values, locales.iterator(), defaultValue);
    }

    public static <Value> Value getLocalized(Map<Locale, Value> values,
            Iterator<Locale> localeIter, Value defaultValue) {
        Value rv = null;

        if (values != null && values.size() > 0) {
            // Use locale search path to find the best matching string
            for (; rv == null && localeIter.hasNext();) {
                rv = values.get(localeIter.next());
            }

            // ...or if no match is found, pick any string
            if (rv == null && !values.isEmpty()) {
                rv = (values.entrySet().iterator().next()).getValue();
            }
        }
        // ...if rv is still null, use defaultValue
        if (rv == null) {
            rv = defaultValue;
        }
        return rv;
    }

    public static <Value> Value getLocalized(Map<Locale, Value> values,
            Locale locale, Value defaultValue) {
        return getLocalized(values, new LocaleIterator(locale, true),
                defaultValue);
    }

    /**
     * @param string
     * @return
     */
    public static Map<Locale, String> parseLiterals(String string) {
        // fi: Suomeksi; en_GB: In english; : default
        Map<Locale, String> rv = new HashMap<Locale, String>();
        String[] literals = string.split(";");
        for (int i = 0; i < literals.length; i++) {
            String[] lsplit = literals[i].split(":", 2);
            if (lsplit.length == 2) {
                rv.put(parseLocale(lsplit[0].trim()), lsplit[1].trim());
            }
        }
        return rv;
    }

    public static String encodeLiterals(Map<Locale, String> literals) {
        Set<Map.Entry<Locale, String>> entries = literals.entrySet();
        StringBuilder sb = new StringBuilder(entries.size() * 10);
        for (Map.Entry<Locale, String> entry : entries) {
            if (sb.length() > 0) {
                sb.append("; ");
            }
            sb.append(entry.getKey());
            sb.append(": ");
            sb.append(entry.getValue());
        }
        return sb.toString();
    }
    
    private static boolean isNotEmpty(String str) {
        return str != null && !str.isEmpty();
    }
}
