/*
 * Copyright (c) 2009 Mysema Ltd.
 * All rights reserved.
 * 
 */
package com.mysema.commons.l10n.support;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Locale;
import java.util.NoSuchElementException;
import java.util.Set;

import com.mysema.commons.l10n.State;
import com.mysema.commons.l10n.States;

/**
 * Used in a wide variety of resource searches. Generates a series of name
 * variations from a base name, a {@link java.util.Locale} and an optional
 * suffix.
 * 
 * @author Samppa Saarela
 * @version $Id: LocalizedNameIterator.java 3009 2008-02-18 17:24:52Z tiwe $
 */

public class LocalizedNameIterator implements Iterator<String> {

    private String suffix;

    private StringBuilder buffer;

    private String baseName;

    private int curstate;

    private Locale locale;

    private Set<String> seen;

    private State[] states;

    private String next;

    private static final State[] DEFAULT_SEARCH_PATH = { States.LCV, States.LC,
            States.L, States.DEFAULT, States.LCV, States.LC, States.L,
            States.BARE };

    public LocalizedNameIterator(String baseName, Locale locale, String suffix) {
        this(baseName, locale, suffix, DEFAULT_SEARCH_PATH);
    }

    /**
     * @param baseName
     * @param locale
     * @param suffix
     */
    public LocalizedNameIterator(String baseName, Locale locale, String suffix,
            State[] states) {
        if (baseName == null) {
            throw new IllegalArgumentException("baseName was null");
        } else if (states == null) {
            throw new IllegalArgumentException("states was null");
        }

        this.locale = locale;
        if (this.locale == null) {
            this.locale = new Locale("");
        }

        this.suffix = suffix;
        if (this.suffix == null) {
            this.suffix = "";
        }

        this.states = states.clone();

        // Prepare baseName, buffer and other variables
        this.baseName = baseName;

        if (this.baseName.endsWith(this.suffix)) {
            // Strip suffix off from baseName
            this.baseName = this.baseName.substring(0, this.baseName.length()
                    - this.suffix.length());
        }

        buffer = new StringBuilder(this.baseName.length() + 16);

        curstate = 0;
        seen = new HashSet<String>((int) (states.length / 0.7));
    }

    /**
     * Returns true if there are more name variants to be returned, false
     * otherwise.
     * 
     */

    public boolean hasNext() {
        advance();
        return next != null;
    }

    /**
     * Returns the next localized variant.
     * 
     * @throws NoSuchElementException
     *             if all variants have been returned.
     * 
     */

    public String next() {
        advance();
        if (next == null) {
            throw new NoSuchElementException();
        }
        String result = next;
        next = null;
        return result;
    }

    private void reset() {
        buffer.replace(0, baseName.length(), baseName);
        buffer.setLength(baseName.length());
    }

    private void advance() {
        if (next == null && curstate < states.length) {
            reset();
            boolean found;
            do {
                // Allow State to switch locale used
                locale = states[curstate].convert(locale);

                // Continue search until
                found = false;
                // 1) some state is able to handle the input,
                found = states[curstate++].apply(locale, buffer);
                // 2) result has not been yet seen and
                if (found && !seen.add(buffer.toString())) {
                    // reset the buffer
                    reset();
                    found = false;
                }
                // 3) there are still alternatives left
            } while (!found && curstate < states.length);

            if (found) {
                buffer.append(suffix);
                next = buffer.toString();
            }
        }
    }

    public Locale getCurrentLocale() {
        if (curstate == 0) {
            throw new NoSuchElementException();
        } else {
            return states[curstate - 1].filter(locale);
        }
    }

    /**
     * @see java.util.Iterator#remove()
     */
    public void remove() {
        throw new UnsupportedOperationException();
    }

}
