/*
 * Decompiled with CFR 0.152.
 */
package net.spy.memcached;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import net.spy.memcached.AddrUtil;
import net.spy.memcached.AdminConnectTimeoutException;
import net.spy.memcached.ArcusClient;
import net.spy.memcached.ArcusClientException;
import net.spy.memcached.ArcusReplNodeAddress;
import net.spy.memcached.CacheMonitor;
import net.spy.memcached.ConnectionFactoryBuilder;
import net.spy.memcached.ConnectionObserver;
import net.spy.memcached.MemcachedConnection;
import net.spy.memcached.NotExistsServiceCodeException;
import net.spy.memcached.compat.SpyThread;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooDefs;
import org.apache.zookeeper.ZooKeeper;

public class CacheManager
extends SpyThread
implements Watcher,
CacheMonitor.CacheMonitorListener {
    private static final String ARCUS_BASE_CACHE_LIST_ZPATH = "/arcus/cache_list/";
    private static final String ARCUS_BASE_CLIENT_INFO_ZPATH = "/arcus/client_list/";
    private static final String ARCUS_REPL_CACHE_LIST_ZPATH = "/arcus_repl/cache_list/";
    private static final String ARCUS_REPL_CLIENT_INFO_ZPATH = "/arcus_repl/client_list/";
    private static final int ZK_SESSION_TIMEOUT = 15000;
    private static final long ZK_CONNECT_TIMEOUT = 15000L;
    private final String hostPort;
    private final String serviceCode;
    private CacheMonitor cacheMonitor;
    private ZooKeeper zk;
    private ArcusClient[] client;
    private final CountDownLatch clientInitLatch;
    private final ConnectionFactoryBuilder cfb;
    private final int waitTimeForConnect;
    private final int poolSize;
    private volatile boolean shutdownRequested = false;
    private CountDownLatch zkInitLatch;
    private List<String> prevCacheList;
    private boolean arcusReplEnabled = false;

    public CacheManager(String hostPort, String serviceCode, ConnectionFactoryBuilder cfb, CountDownLatch clientInitLatch, int poolSize, int waitTimeForConnect) {
        this.hostPort = hostPort;
        this.serviceCode = serviceCode;
        this.cfb = cfb;
        this.clientInitLatch = clientInitLatch;
        this.poolSize = poolSize;
        this.waitTimeForConnect = waitTimeForConnect;
        this.initZooKeeperClient();
        this.setName("Cache Manager IO for " + serviceCode + "@" + hostPort);
        this.setDaemon(true);
        this.start();
        this.getLogger().info("CacheManager started. (" + serviceCode + "@" + hostPort + ")");
    }

    private void initZooKeeperClient() {
        try {
            this.getLogger().info("Trying to connect to Arcus admin(%s@%s)", this.serviceCode, this.hostPort);
            this.zkInitLatch = new CountDownLatch(1);
            this.zk = new ZooKeeper(this.hostPort, 15000, (Watcher)this);
            try {
                if (!this.zkInitLatch.await(15000L, TimeUnit.MILLISECONDS)) {
                    this.getLogger().fatal("Connecting to Arcus admin(%s) timed out : %d miliseconds", this.hostPort, 15000L);
                    throw new AdminConnectTimeoutException(this.hostPort);
                }
                if (this.zk.exists(ARCUS_REPL_CACHE_LIST_ZPATH + this.serviceCode, false) != null) {
                    this.arcusReplEnabled = true;
                    this.cfb.internalArcusReplEnabled(true);
                    this.getLogger().info("Connecting to Arcus repl cluster");
                } else if (this.zk.exists(ARCUS_BASE_CACHE_LIST_ZPATH + this.serviceCode, false) != null) {
                    this.arcusReplEnabled = false;
                    this.cfb.internalArcusReplEnabled(false);
                    this.getLogger().info("Connecting to Arcus cluster");
                } else {
                    this.getLogger().fatal("ARCUS cluster named \"%s\" not found.", this.serviceCode);
                    throw new NotExistsServiceCodeException(this.serviceCode);
                }
                String path = this.getClientInfo();
                if (path.isEmpty()) {
                    this.getLogger().fatal("Can't create the znode of client info (" + path + ")");
                    throw new ArcusClientException.InitializeClientException("Can't create client info");
                }
                if (this.zk.exists(path, false) == null) {
                    this.zk.create(path, null, (List)ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.EPHEMERAL);
                }
            }
            catch (AdminConnectTimeoutException e) {
                this.shutdownZooKeeperClient();
                throw e;
            }
            catch (NotExistsServiceCodeException e) {
                this.shutdownZooKeeperClient();
                throw e;
            }
            catch (ArcusClientException.InitializeClientException e) {
                this.shutdownZooKeeperClient();
                throw e;
            }
            catch (InterruptedException ie) {
                this.getLogger().fatal("Can't connect to Arcus admin(%s@%s) %s", this.serviceCode, this.hostPort, ie.getMessage());
                this.shutdownZooKeeperClient();
                return;
            }
            catch (Exception e) {
                this.getLogger().fatal((Object)"Unexpected exception. contact to Arcus administrator", e);
                this.shutdownZooKeeperClient();
                throw new ArcusClientException.InitializeClientException("Can't initialize Arcus client.", e);
            }
            String cacheListZPath = this.arcusReplEnabled ? ARCUS_REPL_CACHE_LIST_ZPATH : ARCUS_BASE_CACHE_LIST_ZPATH;
            this.cacheMonitor = new CacheMonitor(this.zk, cacheListZPath, this.serviceCode, this);
        }
        catch (IOException e) {
            throw new ArcusClientException.InitializeClientException("Can't initialize Arcus client.", e);
        }
    }

    private String getClientInfo() {
        String restInfo;
        String hostInfo;
        String path = "";
        path = this.arcusReplEnabled ? ARCUS_REPL_CLIENT_INFO_ZPATH + this.serviceCode + "/" : ARCUS_BASE_CLIENT_INFO_ZPATH + this.serviceCode + "/";
        try {
            hostInfo = InetAddress.getLocalHost().getHostName() + "_" + InetAddress.getLocalHost().getHostAddress() + "_";
        }
        catch (Exception e) {
            this.getLogger().fatal((Object)"Can't get client host info.", e);
            hostInfo = "unknown-host_0.0.0.0_";
        }
        path = path + hostInfo + this.poolSize + "_java_" + ArcusClient.getVersion() + "_";
        try {
            SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyyMMddHHmmss");
            Date currentTime = new Date();
            restInfo = simpleDateFormat.format(currentTime) + "_" + this.zk.getSessionId();
        }
        catch (Exception e) {
            this.getLogger().fatal((Object)"Can't get time and zk session id.", e);
            restInfo = "00000000000000_0";
        }
        path = path + restInfo;
        return path;
    }

    public void process(WatchedEvent event) {
        if (event.getType() == Watcher.Event.EventType.None) {
            switch (event.getState()) {
                case SyncConnected: {
                    this.zkInitLatch.countDown();
                    this.getLogger().info("Connected to Arcus admin. (%s@%s)", this.serviceCode, this.hostPort);
                    if (this.cacheMonitor != null) {
                        this.getLogger().warn("Reconnected to the Arcus admin. " + this.getInfo());
                        break;
                    }
                    this.getLogger().debug("cm is null, servicecode : %s, state:%s, type:%s", this.serviceCode, event.getState(), event.getType());
                    break;
                }
                case Disconnected: {
                    this.getLogger().warn("Disconnected from the Arcus admin. Trying to reconnect. " + this.getInfo());
                    break;
                }
                case Expired: {
                    this.getLogger().warn("Session expired. Trying to reconnect to the Arcus admin." + this.getInfo());
                    if (this.cacheMonitor == null) break;
                    this.cacheMonitor.shutdown();
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public void run() {
        CacheManager cacheManager = this;
        synchronized (cacheManager) {
            while (true) {
                if (this.shutdownRequested) {
                    // MONITOREXIT @DISABLED, blocks:[4, 11, 13] lbl5 : MonitorExitStatement: MONITOREXIT : var1_1
                    this.getLogger().info("Close cache manager.");
                    this.shutdownZooKeeperClient();
                    return;
                }
                if (!this.cacheMonitor.isDead()) {
                    try {
                        this.wait();
                    }
                    catch (InterruptedException e) {
                        this.getLogger().warn("Cache mananger thread is interrupted while wait: %s", e.getMessage());
                    }
                    continue;
                }
                long retrySleepTime = 0L;
                try {
                    this.getLogger().warn("Unexpected disconnection from Arcus admin. Trying to reconnect to Arcus admin. CacheList =" + this.prevCacheList);
                    this.shutdownZooKeeperClient();
                    this.initZooKeeperClient();
                }
                catch (AdminConnectTimeoutException e) {
                    retrySleepTime = 1000L;
                }
                catch (NotExistsServiceCodeException e) {
                    retrySleepTime = 5000L;
                }
                catch (ArcusClientException.InitializeClientException e) {
                    retrySleepTime = 5000L;
                }
                catch (Exception e) {
                    retrySleepTime = 1000L;
                    this.getLogger().warn("upexpected exception is caught while reconnet to Arcus admin: %s", e.getMessage());
                }
                if (retrySleepTime <= 0L) continue;
                try {
                    Thread.sleep(retrySleepTime);
                }
                catch (InterruptedException e) {
                    this.getLogger().warn("Cache mananger thread is interrupted while sleep: %s", e.getMessage());
                    continue;
                }
                break;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void closing() {
        CacheManager cacheManager = this;
        synchronized (cacheManager) {
            this.notifyAll();
        }
    }

    private String getAddressListString(List<String> children) {
        StringBuilder addrs = new StringBuilder();
        for (int i = 0; i < children.size(); ++i) {
            String[] temp = children.get(i).split("-");
            if (i != 0) {
                addrs.append(",").append(temp[0]);
                continue;
            }
            addrs.append(temp[0]);
        }
        return addrs.toString();
    }

    @Override
    public void commandCacheListChange(List<String> children) {
        if (children.size() == 0) {
            this.getLogger().error("Cannot find any cache nodes for your service code. Please contact Arcus support to solve this problem. [serviceCode=" + this.serviceCode + ", addminSessionId=0x" + Long.toHexString(this.zk.getSessionId()));
        }
        if (!children.equals(this.prevCacheList)) {
            this.getLogger().warn("Cache list has been changed : From=" + this.prevCacheList + ", " + "To=" + children + ", " + "[serviceCode=" + this.serviceCode + ", addminSessionId=0x" + Long.toHexString(this.zk.getSessionId()));
        }
        this.prevCacheList = children;
        String addrs = this.getAddressListString(children);
        if (this.client == null) {
            this.createArcusClient(addrs);
            return;
        }
        for (ArcusClient ac : this.client) {
            MemcachedConnection conn = ac.getMemcachedConnection();
            conn.putMemcachedQueue(addrs);
        }
    }

    @Override
    public List<String> getPrevCacheList() {
        return this.prevCacheList;
    }

    private String getInfo() {
        String zkSessionId = null;
        if (this.zk != null) {
            zkSessionId = "0x" + Long.toHexString(this.zk.getSessionId());
        }
        return "[serviceCode=" + this.serviceCode + ", adminSessionId=" + zkSessionId + "]";
    }

    private void createArcusClient(String addrs) {
        List<InetSocketAddress> socketList;
        if (this.arcusReplEnabled) {
            socketList = ArcusReplNodeAddress.getAddresses(addrs);
            Map<String, List<ArcusReplNodeAddress>> newAllGroups = ArcusReplNodeAddress.makeGroupAddrsList(socketList);
            socketList.clear();
            for (Map.Entry<String, List<ArcusReplNodeAddress>> entry : newAllGroups.entrySet()) {
                if (entry.getValue().size() <= 0) continue;
                socketList.addAll((Collection<InetSocketAddress>)entry.getValue());
            }
        } else {
            socketList = AddrUtil.getAddresses(addrs);
        }
        int addrCount = socketList.size();
        final CountDownLatch latch = new CountDownLatch(addrCount * this.poolSize);
        ConnectionObserver observer = new ConnectionObserver(){

            @Override
            public void connectionLost(SocketAddress sa) {
            }

            @Override
            public void connectionEstablished(SocketAddress sa, int reconnectCount) {
                latch.countDown();
            }
        };
        this.cfb.setInitialObservers(Collections.singleton(observer));
        int _awaitTime = 0;
        _awaitTime = this.waitTimeForConnect == 0 ? 50 * addrCount * this.poolSize : this.waitTimeForConnect;
        this.client = new ArcusClient[this.poolSize];
        for (int i = 0; i < this.poolSize; ++i) {
            try {
                String clientName = "ArcusClient(" + (i + 1) + "-" + this.poolSize + ") for " + this.serviceCode;
                this.client[i] = ArcusClient.getInstance(this.cfb.build(), clientName, socketList);
                this.client[i].setName("Memcached IO for " + this.serviceCode);
                this.client[i].setCacheManager(this);
                continue;
            }
            catch (IOException e) {
                this.getLogger().fatal("Arcus Connection has critical problems. contact arcus manager.");
            }
        }
        try {
            if (latch.await(_awaitTime, TimeUnit.MILLISECONDS)) {
                this.getLogger().warn("All arcus connections are established.");
            } else {
                this.getLogger().error("Some arcus connections are not established.");
            }
        }
        catch (InterruptedException e) {
            this.getLogger().fatal("Arcus Connection has critical problems. contact arcus manager.");
        }
        this.clientInitLatch.countDown();
    }

    public ArcusClient[] getAC() {
        return this.client;
    }

    private void shutdownZooKeeperClient() {
        if (this.zk == null) {
            return;
        }
        try {
            this.getLogger().info("Close the ZooKeeper client. serviceCode=" + this.serviceCode + ", adminSessionId=0x" + Long.toHexString(this.zk.getSessionId()));
            this.zk.close();
            this.zk = null;
        }
        catch (InterruptedException e) {
            this.getLogger().warn((Object)"An exception occured while closing ZooKeeper client.", e);
        }
    }

    public void shutdown() {
        if (!this.shutdownRequested) {
            this.getLogger().info("Shut down cache manager.");
            this.shutdownRequested = true;
            this.closing();
        }
    }
}

