package com.netease.cloudmusic.iotsdk.sdkbase.base.network.config

import com.netease.cloudmusic.iotsdk.sdkbase.base.storage.mmkv.CMSharedPreferences
import okhttp3.Cookie
import okhttp3.CookieJar
import okhttp3.HttpUrl
import org.json.JSONObject
import java.util.concurrent.ConcurrentHashMap

/**
 * cookie相关管理类
 * created by chenchiyi on 2022/1/20 10:53
 */
object CMCookieStore : CookieJar {

    private const val MMKV_ID = "cookie"
    private const val COOKIE_NAME = "name"
    private const val COOKIE_VALUE = "value"
    private const val COOKIE_PATH = "path"
    private const val COOKIE_SECURE = "secure"
    private const val COOKIE_EXPIRE = "expire"
    private const val COOKIE_DOMAIN = "domain"

    private var mCookies: ConcurrentHashMap<String, Cookie>
    private var cookieFile = CMSharedPreferences.getMultiProcessSettingPreference(MMKV_ID)

    init {
        mCookies = initMemoryCookieAndSyncLoadFromFile()
    }

    private fun initMemoryCookieAndSyncLoadFromFile(): ConcurrentHashMap<String, Cookie> {
        return ConcurrentHashMap()
    }

    override fun saveFromResponse(url: HttpUrl, cookies: MutableList<Cookie>) {
        saveCookies(cookies)
    }

    override fun loadForRequest(url: HttpUrl): MutableList<Cookie> {
        return loadCookie(url)
    }

    @Synchronized
    private fun saveCookies(cookies: MutableList<Cookie>) {
        for (cookie in cookies) {
            if (isCookieExpired(cookie)) {
                removeCookie(cookie)
            } else {
                val cookieToken = getCookieToken(cookie)
                mCookies[cookieToken] = cookie //内存缓存
                cookieFile.edit().putString(cookieToken, encodeCookie(cookie)).apply()
            }
        }
    }

    @Synchronized
    private fun loadCookie(url: HttpUrl): MutableList<Cookie> {
        val ret = ArrayList<Cookie>()
        val it: Iterator<Map.Entry<String, Cookie>> = mCookies.entries.iterator()
        while (it.hasNext()) {
            val cookie = it.next().value
            if (isCookieExpired(cookie)) {
                removeCookie(cookie)
            } else {
                if (cookie.matches(url)) {
                    ret.add(cookie)
                }
            }
        }
        return ret
    }

    /**
     * 当前cookie是否过期
     */
    private fun isCookieExpired(cookie: Cookie): Boolean {
        return cookie.expiresAt() < System.currentTimeMillis()
    }

    /**
     * 根据url移除当前的cookie
     */
    @Synchronized
    fun removeCookie(cookie: Cookie): Boolean {
        val name: String = getCookieToken(cookie)
        return if (mCookies.containsKey(name)) {
            //内存移除
            mCookies.remove(name)
            //文件移除
            cookieFile.edit().remove(name).apply()
            true
        } else {
            false
        }
    }

    private fun encodeCookie(cookie: Cookie): String {
        val jsonObject = JSONObject()
        jsonObject.put(COOKIE_NAME, cookie.name())
        jsonObject.put(COOKIE_VALUE, cookie.value())
        jsonObject.put(COOKIE_EXPIRE, cookie.expiresAt())
        jsonObject.put(COOKIE_DOMAIN, cookie.domain())
        jsonObject.put(COOKIE_PATH, cookie.path())
        jsonObject.put(COOKIE_SECURE, cookie.secure())
        return jsonObject.toString()
    }

    private fun getCookieToken(cookie: Cookie): String {
        return cookie.name() + "_" + cookie.domain() + "_" + cookie.path()
    }
}