package com.netease.cloudmusic.iotsdk.sdkbase.base.network.exception

import androidx.annotation.Keep
import org.json.JSONException
import java.io.IOException
import java.net.*
import javax.net.ssl.SSLException

@Keep
class ApiException @JvmOverloads constructor(
    val code: Int,
    val subCode: Int,
    val msg: String,
    tipErrMsg: String = "$code($subCode):${msg}"
) : CMHttpRequestException(tipErrMsg) {

    interface CODE {
        companion object {
            const val SERVER_ERR = 1
            const val IO_ERR = 2
            const val NOT_FOUND = 3
        }
    }

    constructor(code: Int, subCode: Int, msg: String, t: Throwable?) : this(
        code,
        subCode,
        msg,
        wrapErrorCodeAndException(code, t)
    ) {
        initCause(t)
    }

    companion object {
        fun isNetworkException(t: Throwable?): Boolean {
            return t != null && t is ApiException && t.code == CODE.IO_ERR
        }

        fun isResourceNotFoundException(t: Throwable?): Boolean {
            return t != null && t is ApiException && t.code == CODE.NOT_FOUND
        }

        fun wrapErrorCodeAndException(code: Int, t: Throwable?): String {
            return "code: " + code + (", " + if (t != null) getMsgFromNetworkThrowable(t) else "")
        }

        fun getMsgFromNetworkThrowable(t: Throwable?): String {
            if (t == null) {
                return ""
            }
            val msg: String = if (t is JSONException) {
                "error: json"
            } else if (t is SocketTimeoutException) {
                val eMsg = t.message
                if (eMsg != null && eMsg.contains("connect")) {
                    "error: ct"
                } else {
                    "error: st"
                }
            } else if (t is ConnectException) {
                "error: conn"
            } else if (t is ProtocolException) {
                "error: protocol"
            } else if (t is SocketException) {
                "error: socket"
            } else if (t is SSLException) {
                "error: ssl"
            } else if (t is UnknownHostException) {
                // 避免网络抖动造成太多的偏差¬
                "error: unkonwnhost"
            } else if (t is IOException) {
                "error: io"
            } else {
                "error: unknown"
            }
            return msg
        }
    }
}