package com.netease.cloudmusic.iotsdk.sdkbase.base.network.interceptor

import com.netease.cloudmusic.iotsdk.sdkbase.base.network.config.CMNetworkConfig
import com.netease.cloudmusic.iotsdk.sdkbase.base.network.const.CodeConst
import com.netease.cloudmusic.iotsdk.sdkbase.base.network.retrofit.ApiResult
import com.netease.cloudmusic.iotsdk.sdkbase.utils.CMSDKLogUtils
import com.netease.cloudmusic.iotsdk.sdkbase.utils.MoshiUtils
import com.squareup.moshi.Types
import okhttp3.Interceptor
import okhttp3.Request
import okhttp3.Response
import okio.Buffer
import java.io.IOException
import java.nio.charset.Charset
import java.util.concurrent.locks.ReentrantLock

/**
 * AccessToken过期刷新 拦截器
 * @Author dengyongbiao
 * @Time 2022/12/27 11:17:43
 */
class AccessTokenRefreshInterceptor : Interceptor {
    @Throws(IOException::class)
    override fun intercept(chain: Interceptor.Chain): Response {
        val originRequest: Request = chain.request()
        // step1： 拿到原请求结果
        val originResponse = chain.proceed(originRequest)
        // 原请求地址
        val originReqUrl = originRequest.url().toString()

        if (needCheckAccessTokenRefresh(originReqUrl) &&
            sFetchLoginInfoLock.holdCount == 0
        ) {
            CMSDKLogUtils.i(
                TAG,
                "Before Lock originReqUrl:" + originReqUrl + "， thread:" + Thread.currentThread().id
            )
            // 重入锁保证只有第一个线程可以抢占到
            sFetchLoginInfoLock.lock()
            CMSDKLogUtils.i(
                TAG,
                "Lock originReqUrl:" + originReqUrl + "， thread:" + Thread.currentThread().id
            )
            try {
                // 拿到返回结果
                val responseBody = originResponse.body() ?: return originResponse

                // 设置编码 准备读取返回数据
                // step5：读取返回数据并判断 accessToken 是否失效、如果失效执行刷新操作
                var charset: Charset? = UTF8
                val bufferedSource = responseBody.source()
                bufferedSource.request(Long.MAX_VALUE)
                val buffer: Buffer = bufferedSource.buffer()

                val contentType = responseBody.contentType()
                if (contentType != null) {
                    charset = contentType.charset(UTF8)
                }
                // 编码为空 直接返回
                if (charset == null) {
                    return originResponse;
                }
                // 网络请求返回数据
                val bodyString = buffer.clone().readString(charset)
                // accessToken过期
                if (accessTokenOverdue(bodyString)) {
                    // step6: 请求 accessToken 并保存结果
                    CMNetworkConfig.accessTokenRefreshHandler?.onAccessTokenRefreshHandle()
                    // 重新发起请求
                    return chain.proceed(originRequest)
                }
            } finally {
                CMSDKLogUtils.i(
                    TAG,
                    "Unlock originReqUrl:" + originReqUrl
                        .toString() + "， thread:" + Thread.currentThread().id
                )
                sFetchLoginInfoLock.unlock()
            }
        }

        return originResponse
    }

    private fun needCheckAccessTokenRefresh(url: String): Boolean {
        return CMNetworkConfig.accessTokenRefreshHandler?.needCheckAccessTokenRefresh(url) == true
    }

    private fun accessTokenOverdue(bodyString: String): Boolean {
        val type = Types.newParameterizedType(ApiResult::class.java, Any::class.java)
        val result = MoshiUtils.fromJson<ApiResult<Any>>(bodyString, type)
        if (result?.code == CodeConst.API_ACCESS_TOKEN_OVERDUE) {
            return true
        }
        return false
    }

    companion object {
        private val UTF8 = Charset.forName("UTF-8")
        private const val TAG = "AccessTokenRefreshInterceptor"
        private val sFetchLoginInfoLock = ReentrantLock()
    }
}