package com.netease.cloudmusic.iotsdk.sdkbase.base.network.retrofit.convert

import com.netease.cloudmusic.iotsdk.sdkbase.base.network.exception.ApiException
import com.netease.cloudmusic.iotsdk.sdkbase.base.network.exception.CMNetworkIOException
import com.netease.cloudmusic.iotsdk.sdkbase.base.network.retrofit.ApiResult
import com.squareup.moshi.JsonAdapter
import com.squareup.moshi.JsonDataException
import com.squareup.moshi.JsonReader
import okhttp3.ResponseBody
import okio.ByteString.Companion.decodeHex
import retrofit2.Converter
import java.io.IOException

internal class CMMoshiResponseBodyConverter<T>(private val adapter: JsonAdapter<T>) :
    Converter<ResponseBody, T?> {
    @Throws(IOException::class)
    override fun convert(value: ResponseBody): T? {
        val source = value.source()
        return value.use { value ->
            // Moshi has no document-level API so the responsibility of BOM skipping falls to whatever
            // is delegating to it. Since it's a UTF-8-only library as well we only honor the UTF-8 BOM.
            if (source.rangeEquals(0, UTF8_BOM)) {
                source.skip(UTF8_BOM.size.toLong())
            }
            val reader = JsonReader.of(source)
            var result: Any? = null
            try {
                result = adapter.fromJson(reader)
            } catch (e: JsonDataException) {
                throw CMNetworkIOException(e)
            }
            if (reader.peek() != JsonReader.Token.END_DOCUMENT) {
                throw CMNetworkIOException(JsonDataException("JSON document was not fully consumed."))
            }
            if (result == null) {
                throw CMNetworkIOException(JsonDataException("JSON DATA NULL"))
            }
            if (result is ApiResult<*> && !result.isSuccess()) {
                throw ApiException(result.code, result.subCode, result.message?: "unknownError")
            }
            result
        }
    }

    companion object {
        private val UTF8_BOM = "EFBBBF".decodeHex()
    }
}