/*
 * Decompiled with CFR 0.152.
 */
package conductor.org.elasticsearch.indices.store;

import conductor.org.apache.logging.log4j.LogManager;
import conductor.org.apache.logging.log4j.Logger;
import conductor.org.apache.logging.log4j.message.ParameterizedMessage;
import conductor.org.elasticsearch.cluster.ClusterChangedEvent;
import conductor.org.elasticsearch.cluster.ClusterName;
import conductor.org.elasticsearch.cluster.ClusterState;
import conductor.org.elasticsearch.cluster.ClusterStateListener;
import conductor.org.elasticsearch.cluster.ClusterStateObserver;
import conductor.org.elasticsearch.cluster.metadata.IndexMetaData;
import conductor.org.elasticsearch.cluster.node.DiscoveryNode;
import conductor.org.elasticsearch.cluster.routing.IndexRoutingTable;
import conductor.org.elasticsearch.cluster.routing.IndexShardRoutingTable;
import conductor.org.elasticsearch.cluster.routing.RoutingNode;
import conductor.org.elasticsearch.cluster.routing.RoutingTable;
import conductor.org.elasticsearch.cluster.routing.ShardRouting;
import conductor.org.elasticsearch.cluster.service.ClusterService;
import conductor.org.elasticsearch.common.collect.Tuple;
import conductor.org.elasticsearch.common.inject.Inject;
import conductor.org.elasticsearch.common.io.stream.StreamInput;
import conductor.org.elasticsearch.common.io.stream.StreamOutput;
import conductor.org.elasticsearch.common.settings.Setting;
import conductor.org.elasticsearch.common.settings.Settings;
import conductor.org.elasticsearch.common.unit.TimeValue;
import conductor.org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import conductor.org.elasticsearch.index.IndexService;
import conductor.org.elasticsearch.index.IndexSettings;
import conductor.org.elasticsearch.index.shard.IndexShard;
import conductor.org.elasticsearch.index.shard.IndexShardState;
import conductor.org.elasticsearch.index.shard.ShardId;
import conductor.org.elasticsearch.indices.IndicesService;
import conductor.org.elasticsearch.threadpool.ThreadPool;
import conductor.org.elasticsearch.transport.TransportChannel;
import conductor.org.elasticsearch.transport.TransportException;
import conductor.org.elasticsearch.transport.TransportRequest;
import conductor.org.elasticsearch.transport.TransportRequestHandler;
import conductor.org.elasticsearch.transport.TransportResponse;
import conductor.org.elasticsearch.transport.TransportResponseHandler;
import conductor.org.elasticsearch.transport.TransportService;
import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

public class IndicesStore
implements ClusterStateListener,
Closeable {
    private static final Logger logger = LogManager.getLogger(IndicesStore.class);
    public static final Setting<TimeValue> INDICES_STORE_DELETE_SHARD_TIMEOUT = Setting.positiveTimeSetting("indices.store.delete.shard.timeout", new TimeValue(30L, TimeUnit.SECONDS), Setting.Property.NodeScope);
    public static final String ACTION_SHARD_EXISTS = "internal:index/shard/exists";
    private static final EnumSet<IndexShardState> ACTIVE_STATES = EnumSet.of(IndexShardState.STARTED);
    private final Settings settings;
    private final IndicesService indicesService;
    private final ClusterService clusterService;
    private final TransportService transportService;
    private final ThreadPool threadPool;
    private final Set<ShardId> folderNotFoundCache = new HashSet<ShardId>();
    private TimeValue deleteShardTimeout;

    @Inject
    public IndicesStore(Settings settings, IndicesService indicesService, ClusterService clusterService, TransportService transportService, ThreadPool threadPool) {
        this.settings = settings;
        this.indicesService = indicesService;
        this.clusterService = clusterService;
        this.transportService = transportService;
        this.threadPool = threadPool;
        transportService.registerRequestHandler(ACTION_SHARD_EXISTS, ShardActiveRequest::new, "same", new ShardActiveRequestHandler());
        this.deleteShardTimeout = INDICES_STORE_DELETE_SHARD_TIMEOUT.get(settings);
        if (DiscoveryNode.isDataNode(settings)) {
            clusterService.addListener(this);
        }
    }

    @Override
    public void close() {
        if (DiscoveryNode.isDataNode(this.settings)) {
            this.clusterService.removeListener(this);
        }
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        if (!event.routingTableChanged()) {
            return;
        }
        if (event.state().blocks().disableStatePersistence()) {
            return;
        }
        RoutingTable routingTable = event.state().routingTable();
        this.folderNotFoundCache.removeIf(shardId -> !routingTable.hasIndex(shardId.getIndex()));
        String localNodeId = event.state().nodes().getLocalNodeId();
        RoutingNode localRoutingNode = event.state().getRoutingNodes().node(localNodeId);
        if (localRoutingNode != null) {
            for (ShardRouting routing : localRoutingNode) {
                this.folderNotFoundCache.remove(routing.shardId());
            }
        }
        for (IndexRoutingTable indexRoutingTable : routingTable) {
            block8: for (IndexShardRoutingTable indexShardRoutingTable : indexRoutingTable) {
                IndexSettings indexSettings;
                ShardId shardId2 = indexShardRoutingTable.shardId();
                if (this.folderNotFoundCache.contains(shardId2) || !IndicesStore.shardCanBeDeleted(localNodeId, indexShardRoutingTable)) continue;
                IndexService indexService = this.indicesService.indexService(indexRoutingTable.getIndex());
                if (indexService == null) {
                    IndexMetaData indexMetaData = event.state().getMetaData().getIndexSafe(indexRoutingTable.getIndex());
                    indexSettings = new IndexSettings(indexMetaData, this.settings);
                } else {
                    indexSettings = indexService.getIndexSettings();
                }
                IndicesService.ShardDeletionCheckResult shardDeletionCheckResult = this.indicesService.canDeleteShardContent(shardId2, indexSettings);
                switch (shardDeletionCheckResult) {
                    case FOLDER_FOUND_CAN_DELETE: {
                        this.deleteShardIfExistElseWhere(event.state(), indexShardRoutingTable);
                        continue block8;
                    }
                    case NO_FOLDER_FOUND: {
                        this.folderNotFoundCache.add(shardId2);
                        continue block8;
                    }
                    case NO_LOCAL_STORAGE: {
                        assert (false) : "shard deletion only runs on data nodes which always have local storage";
                        continue block8;
                    }
                    case STILL_ALLOCATED: {
                        continue block8;
                    }
                }
                assert (false) : "unknown shard deletion check result: " + (Object)((Object)shardDeletionCheckResult);
            }
        }
    }

    static boolean shardCanBeDeleted(String localNodeId, IndexShardRoutingTable indexShardRoutingTable) {
        if (indexShardRoutingTable.size() == 0) {
            return false;
        }
        for (ShardRouting shardRouting : indexShardRoutingTable) {
            if (!shardRouting.started()) {
                return false;
            }
            if (!localNodeId.equals(shardRouting.currentNodeId())) continue;
            return false;
        }
        return true;
    }

    private void deleteShardIfExistElseWhere(ClusterState state, IndexShardRoutingTable indexShardRoutingTable) {
        ArrayList<Tuple<DiscoveryNode, ShardActiveRequest>> requests = new ArrayList<Tuple<DiscoveryNode, ShardActiveRequest>>(indexShardRoutingTable.size());
        String indexUUID = indexShardRoutingTable.shardId().getIndex().getUUID();
        ClusterName clusterName = state.getClusterName();
        for (ShardRouting shardRouting : indexShardRoutingTable) {
            assert (shardRouting.started()) : "expected started shard but was " + shardRouting;
            DiscoveryNode discoveryNode = state.nodes().get(shardRouting.currentNodeId());
            requests.add(new Tuple<DiscoveryNode, ShardActiveRequest>(discoveryNode, new ShardActiveRequest(clusterName, indexUUID, shardRouting.shardId(), this.deleteShardTimeout)));
        }
        ShardActiveResponseHandler responseHandler = new ShardActiveResponseHandler(indexShardRoutingTable.shardId(), state.getVersion(), requests.size());
        for (Tuple tuple : requests) {
            logger.trace("{} sending shard active check to {}", (Object)((ShardActiveRequest)tuple.v2()).shardId, tuple.v1());
            this.transportService.sendRequest((DiscoveryNode)tuple.v1(), ACTION_SHARD_EXISTS, (TransportRequest)tuple.v2(), responseHandler);
        }
    }

    private static class ShardActiveResponse
    extends TransportResponse {
        private final boolean shardActive;
        private final DiscoveryNode node;

        ShardActiveResponse(boolean shardActive, DiscoveryNode node) {
            this.shardActive = shardActive;
            this.node = node;
        }

        ShardActiveResponse(StreamInput in) throws IOException {
            this.shardActive = in.readBoolean();
            this.node = new DiscoveryNode(in);
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            out.writeBoolean(this.shardActive);
            this.node.writeTo(out);
        }
    }

    private static class ShardActiveRequest
    extends TransportRequest {
        protected TimeValue timeout = null;
        private ClusterName clusterName;
        private String indexUUID;
        private ShardId shardId;

        ShardActiveRequest() {
        }

        ShardActiveRequest(ClusterName clusterName, String indexUUID, ShardId shardId, TimeValue timeout) {
            this.shardId = shardId;
            this.indexUUID = indexUUID;
            this.clusterName = clusterName;
            this.timeout = timeout;
        }

        @Override
        public void readFrom(StreamInput in) throws IOException {
            super.readFrom(in);
            this.clusterName = new ClusterName(in);
            this.indexUUID = in.readString();
            this.shardId = ShardId.readShardId(in);
            this.timeout = new TimeValue(in.readLong(), TimeUnit.MILLISECONDS);
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            this.clusterName.writeTo(out);
            out.writeString(this.indexUUID);
            this.shardId.writeTo(out);
            out.writeLong(this.timeout.millis());
        }
    }

    private class ShardActiveRequestHandler
    implements TransportRequestHandler<ShardActiveRequest> {
        private ShardActiveRequestHandler() {
        }

        @Override
        public void messageReceived(final ShardActiveRequest request, final TransportChannel channel) throws Exception {
            IndexShard indexShard = this.getShard(request);
            if (indexShard == null) {
                channel.sendResponse(new ShardActiveResponse(false, IndicesStore.this.clusterService.localNode()));
            } else {
                ClusterStateObserver observer = new ClusterStateObserver(IndicesStore.this.clusterService, request.timeout, logger, IndicesStore.this.threadPool.getThreadContext());
                boolean shardActive = this.shardActive(indexShard);
                if (shardActive) {
                    channel.sendResponse(new ShardActiveResponse(true, IndicesStore.this.clusterService.localNode()));
                } else {
                    observer.waitForNextChange(new ClusterStateObserver.Listener(){

                        @Override
                        public void onNewClusterState(ClusterState state) {
                            this.sendResult(ShardActiveRequestHandler.this.shardActive(ShardActiveRequestHandler.this.getShard(request)));
                        }

                        @Override
                        public void onClusterServiceClose() {
                            this.sendResult(false);
                        }

                        @Override
                        public void onTimeout(TimeValue timeout) {
                            this.sendResult(ShardActiveRequestHandler.this.shardActive(ShardActiveRequestHandler.this.getShard(request)));
                        }

                        public void sendResult(boolean shardActive) {
                            try {
                                channel.sendResponse(new ShardActiveResponse(shardActive, IndicesStore.this.clusterService.localNode()));
                            }
                            catch (IOException e) {
                                logger.error(() -> new ParameterizedMessage("failed send response for shard active while trying to delete shard {} - shard will probably not be removed", (Object)request.shardId), (Throwable)e);
                            }
                            catch (EsRejectedExecutionException e) {
                                logger.error(() -> new ParameterizedMessage("failed send response for shard active while trying to delete shard {} - shard will probably not be removed", (Object)request.shardId), (Throwable)e);
                            }
                        }
                    }, newState -> {
                        IndexShard currentShard = this.getShard(request);
                        return currentShard == null || this.shardActive(currentShard);
                    });
                }
            }
        }

        private boolean shardActive(IndexShard indexShard) {
            if (indexShard != null) {
                return ACTIVE_STATES.contains((Object)indexShard.state());
            }
            return false;
        }

        private IndexShard getShard(ShardActiveRequest request) {
            ClusterName thisClusterName = IndicesStore.this.clusterService.getClusterName();
            if (!thisClusterName.equals(request.clusterName)) {
                logger.trace("shard exists request meant for cluster[{}], but this is cluster[{}], ignoring request", (Object)request.clusterName, (Object)thisClusterName);
                return null;
            }
            ShardId shardId = request.shardId;
            IndexService indexService = IndicesStore.this.indicesService.indexService(shardId.getIndex());
            if (indexService != null && indexService.indexUUID().equals(request.indexUUID)) {
                return indexService.getShardOrNull(shardId.id());
            }
            return null;
        }
    }

    private class ShardActiveResponseHandler
    implements TransportResponseHandler<ShardActiveResponse> {
        private final ShardId shardId;
        private final int expectedActiveCopies;
        private final long clusterStateVersion;
        private final AtomicInteger awaitingResponses;
        private final AtomicInteger activeCopies;

        ShardActiveResponseHandler(ShardId shardId, long clusterStateVersion, int expectedActiveCopies) {
            this.shardId = shardId;
            this.expectedActiveCopies = expectedActiveCopies;
            this.clusterStateVersion = clusterStateVersion;
            this.awaitingResponses = new AtomicInteger(expectedActiveCopies);
            this.activeCopies = new AtomicInteger();
        }

        @Override
        public ShardActiveResponse read(StreamInput in) throws IOException {
            return new ShardActiveResponse(in);
        }

        @Override
        public void handleResponse(ShardActiveResponse response) {
            logger.trace("{} is {}active on node {}", (Object)this.shardId, (Object)(response.shardActive ? "" : "not "), (Object)response.node);
            if (response.shardActive) {
                this.activeCopies.incrementAndGet();
            }
            if (this.awaitingResponses.decrementAndGet() == 0) {
                this.allNodesResponded();
            }
        }

        @Override
        public void handleException(TransportException exp) {
            logger.debug(() -> new ParameterizedMessage("shards active request failed for {}", (Object)this.shardId), (Throwable)exp);
            if (this.awaitingResponses.decrementAndGet() == 0) {
                this.allNodesResponded();
            }
        }

        @Override
        public String executor() {
            return "same";
        }

        private void allNodesResponded() {
            if (this.activeCopies.get() != this.expectedActiveCopies) {
                logger.trace("not deleting shard {}, expected {} active copies, but only {} found active copies", (Object)this.shardId, (Object)this.expectedActiveCopies, (Object)this.activeCopies.get());
                return;
            }
            ClusterState latestClusterState = IndicesStore.this.clusterService.state();
            if (this.clusterStateVersion != latestClusterState.getVersion()) {
                logger.trace("not deleting shard {}, the latest cluster state version[{}] is not equal to cluster state before shard active api call [{}]", (Object)this.shardId, (Object)latestClusterState.getVersion(), (Object)this.clusterStateVersion);
                return;
            }
            IndicesStore.this.clusterService.getClusterApplierService().runOnApplierThread("indices_store ([" + this.shardId + "] active fully on other nodes)", currentState -> {
                if (this.clusterStateVersion != currentState.getVersion()) {
                    logger.trace("not deleting shard {}, the update task state version[{}] is not equal to cluster state before shard active api call [{}]", (Object)this.shardId, (Object)currentState.getVersion(), (Object)this.clusterStateVersion);
                    return;
                }
                try {
                    IndicesStore.this.indicesService.deleteShardStore("no longer used", this.shardId, (ClusterState)currentState);
                }
                catch (Exception ex) {
                    logger.debug(() -> new ParameterizedMessage("{} failed to delete unallocated shard, ignoring", (Object)this.shardId), (Throwable)ex);
                }
            }, (source, e) -> logger.error(() -> new ParameterizedMessage("{} unexpected error during deletion of unallocated shard", (Object)this.shardId), (Throwable)e));
        }
    }
}

