/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.curator.framework.recipes.cache;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.netflix.curator.framework.CuratorFramework;
import com.netflix.curator.framework.api.BackgroundCallback;
import com.netflix.curator.framework.api.BackgroundPathable;
import com.netflix.curator.framework.api.CuratorEvent;
import com.netflix.curator.framework.api.GetDataWatchBackgroundStatable;
import com.netflix.curator.framework.api.Pathable;
import com.netflix.curator.framework.api.WatchPathable;
import com.netflix.curator.framework.listen.ListenerContainer;
import com.netflix.curator.framework.recipes.cache.ChildData;
import com.netflix.curator.framework.recipes.cache.PathChildrenCacheEvent;
import com.netflix.curator.framework.recipes.cache.PathChildrenCacheListener;
import com.netflix.curator.framework.recipes.cache.PathChildrenCacheMode;
import com.netflix.curator.framework.state.ConnectionState;
import com.netflix.curator.framework.state.ConnectionStateListener;
import com.netflix.curator.utils.EnsurePath;
import com.netflix.curator.utils.ThreadUtils;
import com.netflix.curator.utils.ZKPaths;
import java.io.Closeable;
import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Exchanger;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PathChildrenCache
implements Closeable {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final CuratorFramework client;
    private final String path;
    private final ExecutorService executorService;
    private final boolean cacheData;
    private final boolean dataIsCompressed;
    private final EnsurePath ensurePath;
    private final Watcher childrenWatcher = new Watcher(){

        public void process(WatchedEvent event) {
            try {
                PathChildrenCache.this.refresh(false);
            }
            catch (Exception e) {
                PathChildrenCache.this.handleException(e);
            }
        }
    };
    private final Watcher dataWatcher = new Watcher(){

        public void process(WatchedEvent event) {
            try {
                if (event.getType() == Watcher.Event.EventType.NodeDeleted) {
                    PathChildrenCache.this.remove(event.getPath());
                } else if (event.getType() == Watcher.Event.EventType.NodeDataChanged) {
                    PathChildrenCache.this.getDataAndStat(event.getPath());
                }
            }
            catch (Exception e) {
                PathChildrenCache.this.handleException(e);
            }
        }
    };
    private final BlockingQueue<PathChildrenCacheEvent> listenerEvents = new LinkedBlockingQueue<PathChildrenCacheEvent>();
    private final ListenerContainer<PathChildrenCacheListener> listeners = new ListenerContainer();
    private final ConcurrentMap<String, ChildData> currentData = Maps.newConcurrentMap();
    @VisibleForTesting
    volatile Exchanger<Object> rebuildTestExchanger;
    private final ConnectionStateListener connectionStateListener = new ConnectionStateListener(){

        public void stateChanged(CuratorFramework client, ConnectionState newState) {
            PathChildrenCache.this.handleStateChange(newState);
        }
    };
    private static final ThreadFactory defaultThreadFactory = ThreadUtils.newThreadFactory((String)"PathChildrenCache");

    public PathChildrenCache(CuratorFramework client, String path, PathChildrenCacheMode mode) {
        this(client, path, mode != PathChildrenCacheMode.CACHE_PATHS_ONLY, false, defaultThreadFactory);
    }

    public PathChildrenCache(CuratorFramework client, String path, PathChildrenCacheMode mode, ThreadFactory threadFactory) {
        this(client, path, mode != PathChildrenCacheMode.CACHE_PATHS_ONLY, false, threadFactory);
    }

    public PathChildrenCache(CuratorFramework client, String path, boolean cacheData) {
        this(client, path, cacheData, false, defaultThreadFactory);
    }

    public PathChildrenCache(CuratorFramework client, String path, boolean cacheData, ThreadFactory threadFactory) {
        this(client, path, cacheData, false, threadFactory);
    }

    public PathChildrenCache(CuratorFramework client, String path, boolean cacheData, boolean dataIsCompressed, ThreadFactory threadFactory) {
        this.client = client;
        this.path = path;
        this.cacheData = cacheData;
        this.dataIsCompressed = dataIsCompressed;
        this.executorService = Executors.newFixedThreadPool(1, threadFactory);
        this.ensurePath = client.newNamespaceAwareEnsurePath(path);
    }

    public void start() throws Exception {
        this.start(false);
    }

    public void start(boolean buildInitial) throws Exception {
        Preconditions.checkState((!this.executorService.isShutdown() ? 1 : 0) != 0, (Object)"already started");
        this.client.getConnectionStateListenable().addListener((Object)this.connectionStateListener);
        this.executorService.submit(new Callable<Object>(){

            @Override
            public Object call() throws Exception {
                PathChildrenCache.this.listenerLoop();
                return null;
            }
        });
        if (buildInitial) {
            this.rebuild();
        } else {
            this.refresh(false);
        }
    }

    public void rebuild() throws Exception {
        Preconditions.checkState((!this.executorService.isShutdown() ? 1 : 0) != 0, (Object)"cache has been closed");
        this.ensurePath.ensure(this.client.getZookeeperClient());
        List children = (List)this.client.getChildren().forPath(this.path);
        for (String child : children) {
            Stat stat;
            String fullPath = ZKPaths.makePath((String)this.path, (String)child);
            if (this.cacheData) {
                try {
                    stat = new Stat();
                    byte[] bytes = this.dataIsCompressed ? (byte[])((WatchPathable)((GetDataWatchBackgroundStatable)this.client.getData().decompressed()).storingStatIn(stat)).forPath(fullPath) : (byte[])((WatchPathable)this.client.getData().storingStatIn(stat)).forPath(fullPath);
                    this.currentData.put(fullPath, new ChildData(fullPath, stat, bytes));
                }
                catch (KeeperException.NoNodeException ignore) {}
            } else {
                stat = (Stat)this.client.checkExists().forPath(fullPath);
                if (stat != null) {
                    this.currentData.put(fullPath, new ChildData(fullPath, stat, null));
                }
            }
            if (this.rebuildTestExchanger == null) continue;
            this.rebuildTestExchanger.exchange(new Object());
        }
        this.refresh(true);
    }

    @Override
    public void close() throws IOException {
        Preconditions.checkState((!this.executorService.isShutdown() ? 1 : 0) != 0, (Object)"has not been started");
        this.client.getConnectionStateListenable().removeListener((Object)this.connectionStateListener);
        this.executorService.shutdownNow();
    }

    public ListenerContainer<PathChildrenCacheListener> getListenable() {
        return this.listeners;
    }

    public List<ChildData> getCurrentData() {
        return ImmutableList.copyOf((Collection)Sets.newTreeSet(this.currentData.values()));
    }

    public ChildData getCurrentData(String fullPath) {
        return (ChildData)this.currentData.get(fullPath);
    }

    public void clearAndRefresh() throws Exception {
        this.currentData.clear();
        this.refresh(false);
    }

    public void clear() {
        this.currentData.clear();
    }

    protected void handleException(Throwable e) {
        this.log.error("", e);
    }

    private void handleStateChange(ConnectionState newState) {
        switch (newState) {
            case SUSPENDED: {
                this.listenerEvents.offer(new PathChildrenCacheEvent(PathChildrenCacheEvent.Type.CONNECTION_SUSPENDED, null));
                break;
            }
            case LOST: {
                this.listenerEvents.offer(new PathChildrenCacheEvent(PathChildrenCacheEvent.Type.CONNECTION_LOST, null));
                break;
            }
            case RECONNECTED: {
                try {
                    this.refresh(true);
                    this.listenerEvents.offer(new PathChildrenCacheEvent(PathChildrenCacheEvent.Type.CONNECTION_RECONNECTED, null));
                    break;
                }
                catch (Exception e) {
                    this.handleException(e);
                }
            }
        }
    }

    private void refresh(final boolean forceGetDataAndStat) throws Exception {
        this.ensurePath.ensure(this.client.getZookeeperClient());
        BackgroundCallback callback = new BackgroundCallback(){

            public void processResult(CuratorFramework client, CuratorEvent event) throws Exception {
                PathChildrenCache.this.processChildren(event.getChildren(), forceGetDataAndStat);
            }
        };
        ((Pathable)((BackgroundPathable)this.client.getChildren().usingWatcher(this.childrenWatcher)).inBackground(callback)).forPath(this.path);
    }

    private void processChildren(List<String> children, boolean forceGetDataAndStat) throws Exception {
        List fullPaths = Lists.transform(children, (Function)new Function<String, String>(){

            public String apply(String child) {
                return ZKPaths.makePath((String)PathChildrenCache.this.path, (String)child);
            }
        });
        HashSet removedNodes = Sets.newHashSet(this.currentData.keySet());
        removedNodes.removeAll(fullPaths);
        for (String fullPath : removedNodes) {
            this.remove(fullPath);
        }
        for (String name : children) {
            String fullPath = ZKPaths.makePath((String)this.path, (String)name);
            if (!forceGetDataAndStat && this.currentData.containsKey(fullPath)) continue;
            this.getDataAndStat(fullPath);
        }
    }

    private void remove(String fullPath) {
        ChildData data = (ChildData)this.currentData.remove(fullPath);
        if (data != null) {
            this.listenerEvents.offer(new PathChildrenCacheEvent(PathChildrenCacheEvent.Type.CHILD_REMOVED, data));
        }
    }

    private void applyNewData(String fullPath, int resultCode, Stat stat, byte[] bytes) {
        if (resultCode == KeeperException.Code.OK.intValue()) {
            ChildData data = new ChildData(fullPath, stat, bytes);
            ChildData previousData = this.currentData.put(fullPath, data);
            if (previousData == null) {
                this.listenerEvents.offer(new PathChildrenCacheEvent(PathChildrenCacheEvent.Type.CHILD_ADDED, data));
            } else if (previousData.getStat().getVersion() != stat.getVersion()) {
                this.listenerEvents.offer(new PathChildrenCacheEvent(PathChildrenCacheEvent.Type.CHILD_UPDATED, data));
            }
        }
    }

    private void getDataAndStat(final String fullPath) throws Exception {
        BackgroundCallback existsCallback = new BackgroundCallback(){

            public void processResult(CuratorFramework client, CuratorEvent event) throws Exception {
                PathChildrenCache.this.applyNewData(fullPath, event.getResultCode(), event.getStat(), null);
            }
        };
        BackgroundCallback getDataCallback = new BackgroundCallback(){

            public void processResult(CuratorFramework client, CuratorEvent event) throws Exception {
                PathChildrenCache.this.applyNewData(fullPath, event.getResultCode(), event.getStat(), event.getData());
            }
        };
        if (this.cacheData) {
            if (this.dataIsCompressed) {
                ((Pathable)((BackgroundPathable)((GetDataWatchBackgroundStatable)this.client.getData().decompressed()).usingWatcher(this.dataWatcher)).inBackground(getDataCallback)).forPath(fullPath);
            } else {
                ((Pathable)((BackgroundPathable)this.client.getData().usingWatcher(this.dataWatcher)).inBackground(getDataCallback)).forPath(fullPath);
            }
        } else {
            ((Pathable)((BackgroundPathable)this.client.checkExists().usingWatcher(this.dataWatcher)).inBackground(existsCallback)).forPath(fullPath);
        }
    }

    private void listenerLoop() {
        while (!Thread.currentThread().isInterrupted()) {
            try {
                PathChildrenCacheEvent event = this.listenerEvents.take();
                this.callListeners(event);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                break;
            }
        }
    }

    private void callListeners(final PathChildrenCacheEvent event) {
        this.listeners.forEach((Function)new Function<PathChildrenCacheListener, Void>(){

            public Void apply(PathChildrenCacheListener listener) {
                try {
                    listener.childEvent(PathChildrenCache.this.client, event);
                }
                catch (Exception e) {
                    PathChildrenCache.this.handleException(e);
                }
                return null;
            }
        });
    }
}

