/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.dyno.recipes.lock;

import com.netflix.discovery.EurekaClient;
import com.netflix.dyno.connectionpool.ConnectionPool;
import com.netflix.dyno.connectionpool.ConnectionPoolConfiguration;
import com.netflix.dyno.connectionpool.ConnectionPoolMonitor;
import com.netflix.dyno.connectionpool.Host;
import com.netflix.dyno.connectionpool.HostSupplier;
import com.netflix.dyno.connectionpool.TokenMapSupplier;
import com.netflix.dyno.connectionpool.impl.ConnectionPoolConfigurationImpl;
import com.netflix.dyno.contrib.ArchaiusConnectionPoolConfiguration;
import com.netflix.dyno.contrib.DynoCPMonitor;
import com.netflix.dyno.contrib.DynoOPMonitor;
import com.netflix.dyno.jedis.DynoJedisClient;
import com.netflix.dyno.jedis.DynoJedisUtils;
import com.netflix.dyno.recipes.lock.LockResource;
import com.netflix.dyno.recipes.lock.VotingHostsFromTokenRange;
import com.netflix.dyno.recipes.lock.VotingHostsSelector;
import com.netflix.dyno.recipes.lock.command.CheckAndRunHost;
import com.netflix.dyno.recipes.lock.command.ExtendHost;
import com.netflix.dyno.recipes.lock.command.LockHost;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DynoLockClient {
    private static final Logger logger = LoggerFactory.getLogger(DynoJedisClient.class);
    private final ConnectionPool pool;
    private final VotingHostsSelector votingHostsSelector;
    private final ExecutorService service;
    private final int quorum;
    private final double CLOCK_DRIFT = 0.01;
    private TimeUnit timeoutUnit;
    private long timeout;
    private final ConcurrentHashMap<String, String> resourceKeyMap = new ConcurrentHashMap();

    public DynoLockClient(ConnectionPool pool, VotingHostsSelector votingHostsSelector, long timeout, TimeUnit unit) {
        this.pool = pool;
        this.votingHostsSelector = votingHostsSelector;
        this.service = Executors.newCachedThreadPool();
        this.quorum = votingHostsSelector.getVotingSize() / 2 + 1;
        this.timeout = timeout;
        this.timeoutUnit = unit;
        Runtime.getRuntime().addShutdownHook(new Thread(() -> this.cleanup()));
    }

    public void setTimeoutUnit(TimeUnit timeoutUnit) {
        this.timeoutUnit = timeoutUnit;
    }

    public void setTimeout(long timeout) {
        this.timeout = timeout;
    }

    private static String getRandomString() {
        return UUID.randomUUID().toString();
    }

    public List<String> getLockedResources() {
        return new ArrayList<String>(this.resourceKeyMap.keySet());
    }

    public void releaseLock(String resource) {
        if (!this.checkResourceExists(resource)) {
            logger.info("No lock held on {}", (Object)resource);
            return;
        }
        CountDownLatch latch = new CountDownLatch(this.votingHostsSelector.getVotingSize());
        this.votingHostsSelector.getVotingHosts().getEntireList().stream().map(host -> new CheckAndRunHost((Host)host, this.pool, "del", resource, this.resourceKeyMap.get(resource))).forEach(ulH -> CompletableFuture.supplyAsync(ulH, this.service).thenAccept(result -> latch.countDown()));
        boolean latchValue = false;
        try {
            latchValue = latch.await(this.timeout, this.timeoutUnit);
        }
        catch (InterruptedException e) {
            logger.info("Interrupted while releasing the lock for resource {}", (Object)resource);
        }
        if (latchValue) {
            logger.info("Released lock on {}", (Object)resource);
        } else {
            logger.info("Timed out before we could release the lock");
        }
        this.resourceKeyMap.remove(resource);
    }

    private TimerTask getExtensionTask(final Timer runJob, final String resource, final long ttlMS, final Consumer<String> extensionFailed) {
        return new TimerTask(){

            @Override
            public void run() {
                long extendedValue = DynoLockClient.this.extendLock(resource, ttlMS);
                if (extendedValue > 0L) {
                    logger.info("Extended lock on {} for {} MS", (Object)resource, (Object)ttlMS);
                    TimerTask task = DynoLockClient.this.getExtensionTask(runJob, resource, ttlMS, extensionFailed);
                    runJob.schedule(task, extendedValue / 2L);
                    return;
                }
                extensionFailed.accept(resource);
            }
        };
    }

    public boolean acquireLock(String resource, long ttlMS, Consumer<String> failure) {
        return this.acquireLockWithExtension(resource, ttlMS, r -> {
            this.releaseLock((String)r);
            failure.accept((String)r);
        });
    }

    private boolean acquireLockWithExtension(String resource, long ttlMS, Consumer<String> extensionFailedCallback) {
        long acquireResult = this.acquireLock(resource, ttlMS);
        if (acquireResult > 0L) {
            Timer runJob = new Timer(resource, true);
            runJob.schedule(this.getExtensionTask(runJob, resource, ttlMS, extensionFailedCallback), acquireResult / 2L);
            return true;
        }
        return false;
    }

    private long runLockHost(String resource, long ttlMS, boolean extend) {
        long startTime = Instant.now().toEpochMilli();
        long drift = Math.round((double)ttlMS * 0.01) + 2L;
        LockResource lockResource = new LockResource(resource, ttlMS);
        CountDownLatch latch = new CountDownLatch(this.quorum);
        if (extend) {
            this.votingHostsSelector.getVotingHosts().getEntireList().stream().map(host -> new ExtendHost((Host)host, this.pool, lockResource, latch, this.resourceKeyMap.get(resource))).forEach(lH -> CompletableFuture.supplyAsync(lH, this.service));
        } else {
            this.votingHostsSelector.getVotingHosts().getEntireList().stream().map(host -> new LockHost((Host)host, this.pool, lockResource, latch, this.resourceKeyMap.get(resource))).forEach(lH -> CompletableFuture.supplyAsync(lH, this.service));
        }
        this.awaitLatch(latch, resource);
        long validity = 0L;
        if (lockResource.getLocked() >= this.quorum) {
            long timeElapsed = Instant.now().toEpochMilli() - startTime;
            validity = ttlMS - timeElapsed - drift;
        } else {
            this.releaseLock(resource);
        }
        return validity;
    }

    public long acquireLock(String resource, long ttlMS) {
        this.resourceKeyMap.putIfAbsent(resource, DynoLockClient.getRandomString());
        return this.runLockHost(resource, ttlMS, false);
    }

    boolean checkResourceExists(String resource) {
        if (!this.resourceKeyMap.containsKey(resource)) {
            logger.info("No lock held on {}", (Object)resource);
            return false;
        }
        return true;
    }

    private boolean awaitLatch(CountDownLatch latch, String resource) {
        try {
            return latch.await(this.timeout, this.timeoutUnit);
        }
        catch (InterruptedException e) {
            logger.info("Interrupted while checking the lock for resource {}", (Object)resource);
            return false;
        }
    }

    public long checkLock(String resource) {
        if (!this.checkResourceExists(resource)) {
            return 0L;
        }
        long startTime = Instant.now().toEpochMilli();
        CopyOnWriteArrayList resultTtls = new CopyOnWriteArrayList();
        CountDownLatch latch = new CountDownLatch(this.quorum);
        this.votingHostsSelector.getVotingHosts().getEntireList().stream().map(host -> new CheckAndRunHost((Host)host, this.pool, "pttl", resource, this.resourceKeyMap.get(resource))).forEach(checkAndRunHost -> CompletableFuture.supplyAsync(checkAndRunHost, this.service).thenAccept(r -> {
            String result = r.getResult().toString();
            if (result.equals("0") || result.equals("-2")) {
                logger.info("Lock not present on host");
            } else {
                resultTtls.add(Long.valueOf(result));
                latch.countDown();
            }
        }));
        boolean latchValue = this.awaitLatch(latch, resource);
        if (latchValue) {
            long timeElapsed = Instant.now().toEpochMilli() - startTime;
            logger.info("Checked lock on {}", (Object)resource);
            return (Long)Collections.min(resultTtls) - timeElapsed;
        }
        logger.info("Timed out before we could check the lock");
        return 0L;
    }

    public long extendLock(String resource, long ttlMS) {
        if (!this.checkResourceExists(resource)) {
            logger.info("Could not extend lock since its already released");
            return 0L;
        }
        return this.runLockHost(resource, ttlMS, true);
    }

    public void cleanup() {
        this.resourceKeyMap.keySet().stream().forEach(this::releaseLock);
    }

    public void logLocks() {
        this.resourceKeyMap.entrySet().stream().forEach(e -> logger.info("Resource: {}, Key: {}", e.getKey(), e.getValue()));
    }

    public static class Builder {
        private String appName;
        private String clusterName;
        private TokenMapSupplier tokenMapSupplier;
        private HostSupplier hostSupplier;
        private ConnectionPoolConfigurationImpl cpConfig;
        private EurekaClient eurekaClient;
        private long timeout;
        private TimeUnit timeoutUnit;

        public Builder withTimeout(long timeout) {
            this.timeout = timeout;
            return this;
        }

        public Builder withTimeoutUnit(TimeUnit unit) {
            this.timeoutUnit = unit;
            return this;
        }

        public Builder withEurekaClient(EurekaClient eurekaClient) {
            this.eurekaClient = eurekaClient;
            return this;
        }

        public Builder withApplicationName(String applicationName) {
            this.appName = applicationName;
            return this;
        }

        public Builder withDynomiteClusterName(String cluster) {
            this.clusterName = cluster;
            return this;
        }

        public Builder withHostSupplier(HostSupplier hSupplier) {
            this.hostSupplier = hSupplier;
            return this;
        }

        public Builder withTokenMapSupplier(TokenMapSupplier tokenMapSupplier) {
            this.tokenMapSupplier = tokenMapSupplier;
            return this;
        }

        public Builder withConnectionPoolConfiguration(ConnectionPoolConfigurationImpl cpConfig) {
            this.cpConfig = cpConfig;
            return this;
        }

        public DynoLockClient build() {
            assert (this.appName != null);
            assert (this.clusterName != null);
            if (this.cpConfig == null) {
                this.cpConfig = new ArchaiusConnectionPoolConfiguration(this.appName);
                logger.info("Dyno Client runtime properties: " + this.cpConfig.toString());
            }
            this.cpConfig.setFallbackEnabled(false);
            this.cpConfig.setConnectToDatastore(true);
            return this.buildDynoLockClient();
        }

        private DynoLockClient buildDynoLockClient() {
            DynoOPMonitor opMonitor = new DynoOPMonitor(this.appName);
            DynoCPMonitor cpMonitor = new DynoCPMonitor(this.appName);
            DynoJedisUtils.updateConnectionPoolConfig((ConnectionPoolConfigurationImpl)this.cpConfig, (HostSupplier)this.hostSupplier, (TokenMapSupplier)this.tokenMapSupplier, (EurekaClient)this.eurekaClient, (String)this.clusterName);
            if (this.tokenMapSupplier == null) {
                this.tokenMapSupplier = this.cpConfig.getTokenSupplier();
            }
            ConnectionPool pool = DynoJedisUtils.createConnectionPool((String)this.appName, (DynoOPMonitor)opMonitor, (ConnectionPoolMonitor)cpMonitor, (ConnectionPoolConfiguration)this.cpConfig, null);
            VotingHostsFromTokenRange votingHostSelector = new VotingHostsFromTokenRange(this.hostSupplier, this.tokenMapSupplier, this.cpConfig.getLockVotingSize());
            return new DynoLockClient(pool, votingHostSelector, this.timeout, this.timeoutUnit);
        }
    }
}

