/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.genie.web.tasks.job;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import com.netflix.genie.common.dto.BaseDTO;
import com.netflix.genie.common.dto.Job;
import com.netflix.genie.common.dto.JobRequest;
import com.netflix.genie.common.dto.JobStatus;
import com.netflix.genie.common.exceptions.GenieException;
import com.netflix.genie.common.exceptions.GenieServerException;
import com.netflix.genie.core.events.JobFinishedEvent;
import com.netflix.genie.core.events.JobFinishedReason;
import com.netflix.genie.core.jobs.JobConstants;
import com.netflix.genie.core.jobs.JobDoneFile;
import com.netflix.genie.core.jobs.JobKillReasonFile;
import com.netflix.genie.core.properties.JobsProperties;
import com.netflix.genie.core.services.JobPersistenceService;
import com.netflix.genie.core.services.JobSearchService;
import com.netflix.genie.core.services.MailService;
import com.netflix.genie.core.services.impl.GenieFileTransferService;
import com.netflix.spectator.api.Counter;
import com.netflix.spectator.api.Id;
import com.netflix.spectator.api.Registry;
import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import javax.validation.constraints.NotNull;
import org.apache.commons.exec.CommandLine;
import org.apache.commons.exec.DefaultExecutor;
import org.apache.commons.exec.ExecuteStreamHandler;
import org.apache.commons.exec.Executor;
import org.apache.commons.exec.PumpStreamHandler;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.io.Resource;
import org.springframework.retry.support.RetryTemplate;
import org.springframework.stereotype.Service;

@Service
public class JobCompletionService {
    private static final Logger log = LoggerFactory.getLogger(JobCompletionService.class);
    private static final String STATUS_TAG = "status";
    private static final String ERROR_TAG = "error";
    private final JobPersistenceService jobPersistenceService;
    private final JobSearchService jobSearchService;
    private final GenieFileTransferService genieFileTransferService;
    private final File baseWorkingDir;
    private final MailService mailServiceImpl;
    private final Executor executor;
    private final boolean deleteArchiveFile;
    private final boolean deleteDependencies;
    private final boolean runAsUserEnabled;
    private final Registry registry;
    private final Id jobCompletionId;
    private final Counter emailSuccessRate;
    private final Counter emailFailureRate;
    private final Counter archivalFailureRate;
    private final Counter doneFileProcessingFailureRate;
    private final Counter finalStatusUpdateFailureRate;
    private final Counter processGroupCleanupFailureRate;
    private final Counter archiveFileDeletionFailure;
    private final Counter deleteDependenciesFailure;
    private final RetryTemplate retryTemplate;
    private final ObjectMapper objectMapper = new ObjectMapper();

    @Autowired
    public JobCompletionService(JobPersistenceService jobPersistenceService, JobSearchService jobSearchService, GenieFileTransferService genieFileTransferService, @Qualifier(value="jobsDir") Resource genieWorkingDir, MailService mailServiceImpl, Registry registry, JobsProperties jobsProperties, @Qualifier(value="genieRetryTemplate") @NotNull RetryTemplate retryTemplate) throws GenieException {
        this.jobPersistenceService = jobPersistenceService;
        this.jobSearchService = jobSearchService;
        this.genieFileTransferService = genieFileTransferService;
        this.mailServiceImpl = mailServiceImpl;
        this.deleteArchiveFile = jobsProperties.getCleanup().isDeleteArchiveFile();
        this.deleteDependencies = jobsProperties.getCleanup().isDeleteDependencies();
        this.runAsUserEnabled = jobsProperties.getUsers().isRunAsUserEnabled();
        this.executor = new DefaultExecutor();
        this.executor.setStreamHandler((ExecuteStreamHandler)new PumpStreamHandler(null, null));
        try {
            this.baseWorkingDir = genieWorkingDir.getFile();
        }
        catch (IOException gse) {
            throw new GenieServerException("Could not load the base path from resource");
        }
        this.registry = registry;
        this.jobCompletionId = registry.createId("genie.jobs.completion.timer");
        this.emailSuccessRate = registry.counter("genie.jobs.email.success.rate");
        this.emailFailureRate = registry.counter("genie.jobs.email.failure.rate");
        this.archivalFailureRate = registry.counter("genie.jobs.archivalFailure.rate");
        this.doneFileProcessingFailureRate = registry.counter("genie.jobs.doneFileProcessingFailure.rate");
        this.finalStatusUpdateFailureRate = registry.counter("genie.jobs.finalStatusUpdateFailure.rate");
        this.processGroupCleanupFailureRate = registry.counter("genie.jobs.processGroupCleanupFailure.rate");
        this.archiveFileDeletionFailure = registry.counter("genie.jobs.archiveFileDeletionFailure.rate");
        this.deleteDependenciesFailure = registry.counter("genie.jobs.deleteDependenciesFailure.rate");
        this.retryTemplate = retryTemplate;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void handleJobCompletion(JobFinishedEvent event) throws GenieException {
        block11: {
            long start = System.nanoTime();
            String jobId = event.getId();
            HashMap tags = Maps.newHashMap();
            try {
                Job job = (Job)this.retryTemplate.execute(context -> this.getJob(jobId));
                JobStatus status = job.getStatus();
                if (!status.isActive()) break block11;
                try {
                    this.retryTemplate.execute(context -> this.updateJob(job, event, tags));
                }
                catch (Exception e) {
                    log.error("Failed updating for job: {}", (Object)jobId, (Object)e);
                    tags.put(ERROR_TAG, "JOB_UPDATE_FAILURE");
                    this.finalStatusUpdateFailureRate.increment();
                }
                try {
                    this.retryTemplate.execute(context -> this.processJobDir(job));
                }
                catch (Exception e) {
                    log.error("Failed archiving directory for job: {}", (Object)jobId, (Object)e);
                    tags.put(ERROR_TAG, "JOB_DIRECTORY_FAILURE");
                    this.archivalFailureRate.increment();
                }
                try {
                    this.retryTemplate.execute(context -> this.sendEmail(jobId));
                }
                catch (Exception e) {
                    log.error("Failed sending email for job: {}", (Object)jobId, (Object)e);
                    tags.put(ERROR_TAG, "SEND_EMAIL_FAILURE");
                    this.emailFailureRate.increment();
                }
            }
            catch (Exception e) {
                log.error("Failed getting job with id: {}", (Object)jobId, (Object)e);
                tags.put(ERROR_TAG, "GET_JOB_FAILURE");
            }
            finally {
                Id timerId = this.jobCompletionId.withTags((Map)tags);
                this.registry.timer(timerId).record(System.nanoTime() - start, TimeUnit.NANOSECONDS);
            }
        }
    }

    private Job getJob(String jobId) throws GenieException {
        return this.jobSearchService.getJob(jobId);
    }

    private Void updateJob(Job job, JobFinishedEvent event, Map<String, String> tags) throws GenieException {
        String jobId = event.getId();
        JobStatus status = job.getStatus();
        JobStatus eventStatus = null;
        if (status == JobStatus.INIT) {
            switch (event.getReason()) {
                case KILLED: {
                    eventStatus = JobStatus.KILLED;
                    break;
                }
                case INVALID: {
                    eventStatus = JobStatus.INVALID;
                    break;
                }
                case FAILED_TO_INIT: {
                    eventStatus = JobStatus.FAILED;
                    break;
                }
                case PROCESS_COMPLETED: {
                    eventStatus = JobStatus.SUCCEEDED;
                    break;
                }
                case SYSTEM_CRASH: {
                    eventStatus = JobStatus.FAILED;
                    break;
                }
                default: {
                    eventStatus = JobStatus.INVALID;
                    log.warn("Unknown event status for job: {}", (Object)jobId);
                    break;
                }
            }
        } else if (event.getReason() != JobFinishedReason.SYSTEM_CRASH) {
            try {
                String finalStatus = (String)this.retryTemplate.execute(context -> this.updateFinalStatusForJob(jobId).toString());
                tags.put(STATUS_TAG, finalStatus);
                this.cleanupProcesses(jobId);
            }
            catch (Exception e) {
                tags.put(ERROR_TAG, "JOB_UPDATE_FINAL_STATUS_FAILURE");
                log.error("Failed updating the exit code and status for job: {}", (Object)jobId, (Object)e);
                this.finalStatusUpdateFailureRate.increment();
            }
        } else {
            eventStatus = JobStatus.FAILED;
        }
        if (eventStatus != null) {
            this.jobPersistenceService.updateJobStatus(jobId, eventStatus, event.getMessage());
            tags.put(STATUS_TAG, eventStatus.toString());
        }
        return null;
    }

    private void cleanupProcesses(String jobId) {
        try {
            if (!this.jobSearchService.getJobStatus(jobId).equals((Object)JobStatus.INVALID)) {
                this.jobSearchService.getJobExecution(jobId).getProcessId().ifPresent(pid -> {
                    try {
                        CommandLine commandLine = new CommandLine("pkill");
                        commandLine.addArgument(JobConstants.getKillFlag());
                        commandLine.addArgument(Integer.toString(pid));
                        this.executor.execute(commandLine);
                        this.processGroupCleanupFailureRate.increment();
                    }
                    catch (Exception e) {
                        log.debug("Received expected exception. Ignoring.");
                    }
                });
            }
        }
        catch (GenieException ge) {
            log.error("Unable to cleanup process for job due to exception. {}", (Object)jobId, (Object)ge);
            this.processGroupCleanupFailureRate.increment();
        }
    }

    private JobStatus updateFinalStatusForJob(String id) throws GenieException {
        log.debug("Updating the status of the job.");
        try {
            JobStatus finalStatus;
            File jobDir = new File(this.baseWorkingDir, id);
            JobDoneFile jobDoneFile = (JobDoneFile)this.objectMapper.readValue(new File(this.baseWorkingDir + "/" + id + "/" + "genie/genie.done"), JobDoneFile.class);
            File killReasonFile = new File(this.baseWorkingDir + "/" + id + "/" + "genie/kill-reason");
            String killedStatusMessages = killReasonFile.exists() ? ((JobKillReasonFile)this.objectMapper.readValue(killReasonFile, JobKillReasonFile.class)).getKillReason() : "Job was killed by user.";
            int exitCode = jobDoneFile.getExitCode();
            File stdOut = new File(jobDir, "stdout");
            Long stdOutSize = stdOut.exists() && stdOut.isFile() ? Long.valueOf(stdOut.length()) : null;
            File stdErr = new File(jobDir, "stderr");
            Long stdErrSize = stdErr.exists() && stdErr.isFile() ? Long.valueOf(stdErr.length()) : null;
            switch (exitCode) {
                case 999: {
                    this.jobPersistenceService.setJobCompletionInformation(id, exitCode, JobStatus.KILLED, killedStatusMessages, stdOutSize, stdErrSize);
                    finalStatus = JobStatus.KILLED;
                    break;
                }
                case 0: {
                    this.jobPersistenceService.setJobCompletionInformation(id, exitCode, JobStatus.SUCCEEDED, "Job finished successfully.", stdOutSize, stdErrSize);
                    finalStatus = JobStatus.SUCCEEDED;
                    break;
                }
                default: {
                    this.jobPersistenceService.setJobCompletionInformation(id, exitCode, JobStatus.FAILED, "Job failed.", stdOutSize, stdErrSize);
                    finalStatus = JobStatus.FAILED;
                }
            }
            return finalStatus;
        }
        catch (IOException ioe) {
            this.doneFileProcessingFailureRate.increment();
            log.error("Could not load the done file for job {}. Marking it as failed.", (Object)id, (Object)ioe);
            this.jobPersistenceService.updateJobStatus(id, JobStatus.FAILED, "Failed to load done file.");
            return JobStatus.FAILED;
        }
    }

    private void deleteApplicationDependencies(String jobId, File jobDir) {
        log.debug("Deleting dependencies as its enabled.");
        if (jobDir.exists()) {
            try {
                List appIds = this.jobSearchService.getJobApplications(jobId).stream().map(BaseDTO::getId).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toList());
                for (String appId : appIds) {
                    File appDependencyDir = new File(jobDir, "genie/applications/" + appId + "/" + "dependencies");
                    this.deleteDependenciesDirectory(appDependencyDir);
                }
            }
            catch (Exception e) {
                log.error("Could not delete job dependencies after completion for job: {} due to error {}", (Object)jobId, (Object)e);
                this.deleteDependenciesFailure.increment();
            }
        }
    }

    private void deleteClusterDependencies(String jobId, File jobDir) {
        log.debug("Deleting dependencies as its enabled.");
        if (jobDir.exists()) {
            try {
                String clusterId = (String)this.jobSearchService.getJobCluster(jobId).getId().orElseThrow(IllegalStateException::new);
                File clusterDependencyDir = new File(jobDir, "genie/cluster/" + clusterId + "/" + "dependencies");
                this.deleteDependenciesDirectory(clusterDependencyDir);
            }
            catch (Exception e) {
                log.error("Could not delete job dependencies after completion for job: {} due to error {}", (Object)jobId, (Object)e);
                this.deleteDependenciesFailure.increment();
            }
        }
    }

    private void deleteDependenciesDirectory(File dependencyDirectory) throws IOException {
        if (dependencyDirectory.exists()) {
            if (this.runAsUserEnabled) {
                CommandLine deleteCommand = new CommandLine("sudo");
                deleteCommand.addArgument("rm");
                deleteCommand.addArgument("-rf");
                deleteCommand.addArgument(dependencyDirectory.getCanonicalPath());
                log.debug("Delete command is {}", (Object)deleteCommand);
                this.executor.execute(deleteCommand);
            } else {
                FileUtils.deleteDirectory((File)dependencyDirectory);
            }
        }
    }

    private boolean processJobDir(Job job) throws GenieException, IOException {
        String jobId;
        File jobDir;
        log.debug("Got a job finished event. Will process job directory.");
        boolean result = false;
        Optional oJobId = job.getId();
        if (oJobId.isPresent() && !this.jobSearchService.getJobStatus((String)job.getId().get()).equals((Object)JobStatus.INVALID) && (jobDir = new File(this.baseWorkingDir, jobId = (String)oJobId.get())).exists()) {
            Optional archiveLocation;
            if (this.deleteDependencies) {
                this.deleteApplicationDependencies(jobId, jobDir);
                this.deleteClusterDependencies(jobId, jobDir);
            }
            if ((archiveLocation = job.getArchiveLocation()).isPresent() && !Strings.isNullOrEmpty((String)((String)archiveLocation.get()))) {
                CommandLine commandLine;
                log.debug("Archiving job directory");
                File localArchiveFile = new File(jobDir, "genie/logs/" + jobId + ".tar.gz");
                if (this.runAsUserEnabled) {
                    commandLine = new CommandLine("sudo");
                    commandLine.addArgument("tar");
                } else {
                    commandLine = new CommandLine("tar");
                }
                commandLine.addArgument("-c");
                commandLine.addArgument("-z");
                commandLine.addArgument("-f");
                commandLine.addArgument(localArchiveFile.getCanonicalPath());
                commandLine.addArgument("./");
                this.executor.setWorkingDirectory(jobDir);
                log.debug("Archive command : {}", (Object)commandLine);
                this.executor.execute(commandLine);
                this.genieFileTransferService.putFile(localArchiveFile.getCanonicalPath(), (String)archiveLocation.get());
                if (this.deleteArchiveFile) {
                    log.debug("Deleting archive file");
                    try {
                        if (this.runAsUserEnabled) {
                            CommandLine deleteCommand = new CommandLine("sudo");
                            deleteCommand.addArgument("rm");
                            deleteCommand.addArgument("-f");
                            deleteCommand.addArgument(localArchiveFile.getCanonicalPath());
                            this.executor.setWorkingDirectory(jobDir);
                            log.debug("Delete command: {}", (Object)deleteCommand);
                            this.executor.execute(deleteCommand);
                        } else if (!localArchiveFile.delete()) {
                            log.error("Failed to delete archive file for job: {}", (Object)jobId);
                            this.archiveFileDeletionFailure.increment();
                        }
                    }
                    catch (Exception e) {
                        log.error("Failed to delete archive file for job: {}", (Object)jobId, (Object)e);
                        this.archiveFileDeletionFailure.increment();
                    }
                }
                result = true;
            }
        }
        return result;
    }

    private boolean sendEmail(String jobId) throws GenieException {
        JobRequest jobRequest = this.jobSearchService.getJobRequest(jobId);
        boolean result = false;
        Optional email = jobRequest.getEmail();
        if (email.isPresent() && !Strings.isNullOrEmpty((String)((String)email.get()))) {
            log.debug("Got a job finished event. Sending email: {}", email.get());
            JobStatus status = this.jobSearchService.getJobStatus(jobId);
            StringBuilder subject = new StringBuilder().append("Genie Job Finished. Id: [").append(jobId).append("], Name: [").append(jobRequest.getName()).append("], Status: [").append(status).append("].");
            StringBuilder body = new StringBuilder().append("Id: [" + jobId + "]\n").append("Name: [" + jobRequest.getName() + "]\n").append("Status: [" + status + "]\n").append("User: [" + jobRequest.getUser() + "]\n").append("Tags: " + jobRequest.getTags() + "\n");
            jobRequest.getDescription().ifPresent(description -> body.append("[" + description + "]"));
            this.mailServiceImpl.sendEmail((String)email.get(), subject.toString(), body.toString());
            result = true;
            this.emailSuccessRate.increment();
        }
        return result;
    }
}

