/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.genie.web.tasks.job;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import com.netflix.genie.common.dto.BaseDTO;
import com.netflix.genie.common.dto.Job;
import com.netflix.genie.common.dto.JobRequest;
import com.netflix.genie.common.dto.JobStatus;
import com.netflix.genie.common.exceptions.GenieException;
import com.netflix.genie.common.exceptions.GenieServerException;
import com.netflix.genie.core.events.JobFinishedEvent;
import com.netflix.genie.core.events.JobFinishedReason;
import com.netflix.genie.core.jobs.JobConstants;
import com.netflix.genie.core.jobs.JobDoneFile;
import com.netflix.genie.core.jobs.JobKillReasonFile;
import com.netflix.genie.core.properties.JobsProperties;
import com.netflix.genie.core.services.JobPersistenceService;
import com.netflix.genie.core.services.JobSearchService;
import com.netflix.genie.core.services.MailService;
import com.netflix.genie.core.services.impl.GenieFileTransferService;
import com.netflix.genie.core.util.MetricsUtils;
import com.netflix.spectator.api.Id;
import com.netflix.spectator.api.Registry;
import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import javax.validation.constraints.NotNull;
import org.apache.commons.exec.CommandLine;
import org.apache.commons.exec.DefaultExecutor;
import org.apache.commons.exec.ExecuteStreamHandler;
import org.apache.commons.exec.Executor;
import org.apache.commons.exec.PumpStreamHandler;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.io.Resource;
import org.springframework.retry.support.RetryTemplate;
import org.springframework.stereotype.Service;

@Service
public class JobCompletionService {
    private static final Logger log = LoggerFactory.getLogger(JobCompletionService.class);
    private static final String ERROR_SOURCE_TAG = "error";
    private static final String JOB_FINAL_STATE = "jobFinalState";
    private final JobPersistenceService jobPersistenceService;
    private final JobSearchService jobSearchService;
    private final GenieFileTransferService genieFileTransferService;
    private final File baseWorkingDir;
    private final MailService mailServiceImpl;
    private final Executor executor;
    private final boolean deleteArchiveFile;
    private final boolean deleteDependencies;
    private final boolean runAsUserEnabled;
    private final Registry registry;
    private final Id jobCompletionTimerId;
    private final Id errorCounterId;
    private final RetryTemplate retryTemplate;
    private final ObjectMapper objectMapper = new ObjectMapper();

    @Autowired
    public JobCompletionService(JobPersistenceService jobPersistenceService, JobSearchService jobSearchService, GenieFileTransferService genieFileTransferService, @Qualifier(value="jobsDir") Resource genieWorkingDir, MailService mailServiceImpl, Registry registry, JobsProperties jobsProperties, @Qualifier(value="genieRetryTemplate") @NotNull RetryTemplate retryTemplate) throws GenieException {
        this.jobPersistenceService = jobPersistenceService;
        this.jobSearchService = jobSearchService;
        this.genieFileTransferService = genieFileTransferService;
        this.mailServiceImpl = mailServiceImpl;
        this.deleteArchiveFile = jobsProperties.getCleanup().isDeleteArchiveFile();
        this.deleteDependencies = jobsProperties.getCleanup().isDeleteDependencies();
        this.runAsUserEnabled = jobsProperties.getUsers().isRunAsUserEnabled();
        this.executor = new DefaultExecutor();
        this.executor.setStreamHandler((ExecuteStreamHandler)new PumpStreamHandler(null, null));
        try {
            this.baseWorkingDir = genieWorkingDir.getFile();
        }
        catch (IOException gse) {
            throw new GenieServerException("Could not load the base path from resource");
        }
        this.registry = registry;
        this.jobCompletionTimerId = registry.createId("genie.jobs.completion.timer");
        this.errorCounterId = registry.createId("genie.jobs.errors.count");
        this.retryTemplate = retryTemplate;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void handleJobCompletion(JobFinishedEvent event) throws GenieException {
        block11: {
            long start = System.nanoTime();
            String jobId = event.getId();
            Map tags = MetricsUtils.newSuccessTagsMap();
            try {
                Job job = (Job)this.retryTemplate.execute(context -> this.getJob(jobId));
                JobStatus status = job.getStatus();
                if (!status.isActive()) break block11;
                try {
                    this.retryTemplate.execute(context -> this.updateJob(job, event, tags));
                }
                catch (Exception e) {
                    log.error("Failed updating for job: {}", (Object)jobId, (Object)e);
                }
                try {
                    this.retryTemplate.execute(context -> this.processJobDir(job));
                }
                catch (Exception e) {
                    log.error("Failed archiving directory for job: {}", (Object)jobId, (Object)e);
                    this.incrementErrorCounter("JOB_DIRECTORY_FAILURE", e);
                }
                try {
                    this.retryTemplate.execute(context -> this.sendEmail(jobId));
                }
                catch (Exception e) {
                    log.error("Failed sending email for job: {}", (Object)jobId, (Object)e);
                    this.incrementErrorCounter("JOB_UPDATE_FAILURE", e);
                }
            }
            catch (Exception e) {
                log.error("Failed getting job with id: {}", (Object)jobId, (Object)e);
                MetricsUtils.addFailureTagsWithException((Map)tags, (Throwable)e);
            }
            finally {
                this.registry.timer(this.jobCompletionTimerId.withTags(tags)).record(System.nanoTime() - start, TimeUnit.NANOSECONDS);
            }
        }
    }

    private Job getJob(String jobId) throws GenieException {
        return this.jobSearchService.getJob(jobId);
    }

    private Void updateJob(Job job, JobFinishedEvent event, Map<String, String> tags) throws GenieException {
        try {
            String jobId = event.getId();
            JobStatus status = job.getStatus();
            JobStatus eventStatus = null;
            if (status == JobStatus.INIT) {
                switch (event.getReason()) {
                    case KILLED: {
                        eventStatus = JobStatus.KILLED;
                        break;
                    }
                    case INVALID: {
                        eventStatus = JobStatus.INVALID;
                        break;
                    }
                    case FAILED_TO_INIT: {
                        eventStatus = JobStatus.FAILED;
                        break;
                    }
                    case PROCESS_COMPLETED: {
                        eventStatus = JobStatus.SUCCEEDED;
                        break;
                    }
                    case SYSTEM_CRASH: {
                        eventStatus = JobStatus.FAILED;
                        break;
                    }
                    default: {
                        eventStatus = JobStatus.INVALID;
                        log.warn("Unknown event status for job: {}", (Object)jobId);
                        break;
                    }
                }
            } else if (event.getReason() != JobFinishedReason.SYSTEM_CRASH) {
                try {
                    String finalStatus = (String)this.retryTemplate.execute(context -> this.updateFinalStatusForJob(jobId).toString());
                    tags.put(JOB_FINAL_STATE, finalStatus);
                    this.cleanupProcesses(jobId);
                }
                catch (Exception e) {
                    log.error("Failed updating the exit code and status for job: {}", (Object)jobId, (Object)e);
                }
            } else {
                tags.put(JOB_FINAL_STATE, JobStatus.FAILED.toString());
                eventStatus = JobStatus.FAILED;
            }
            if (eventStatus != null) {
                tags.put(JOB_FINAL_STATE, status.toString());
                this.jobPersistenceService.updateJobStatus(jobId, eventStatus, event.getMessage());
            }
        }
        catch (Throwable t) {
            this.incrementErrorCounter("JOB_UPDATE_FAILURE", t);
            throw t;
        }
        return null;
    }

    private void cleanupProcesses(String jobId) {
        try {
            if (!this.jobSearchService.getJobStatus(jobId).equals((Object)JobStatus.INVALID)) {
                this.jobSearchService.getJobExecution(jobId).getProcessId().ifPresent(pid -> {
                    try {
                        CommandLine commandLine = new CommandLine("pkill");
                        commandLine.addArgument(JobConstants.getKillFlag());
                        commandLine.addArgument(Integer.toString(pid));
                        this.executor.execute(commandLine);
                        this.incrementErrorCounter("JOB_PROCESS_CLEANUP_NOT_THROWING_FAILURE", new RuntimeException());
                    }
                    catch (Exception e) {
                        log.debug("Received expected exception. Ignoring.");
                    }
                });
            }
        }
        catch (GenieException ge) {
            log.error("Unable to cleanup process for job due to exception. {}", (Object)jobId, (Object)ge);
            this.incrementErrorCounter("JOB_CLEANUP_FAILURE", ge);
        }
        catch (Throwable t) {
            this.incrementErrorCounter("JOB_PROCESS_CLEANUP_FAILURE", t);
            throw t;
        }
    }

    private JobStatus updateFinalStatusForJob(String id) throws GenieException {
        log.debug("Updating the status of the job.");
        try {
            JobStatus finalStatus;
            File jobDir = new File(this.baseWorkingDir, id);
            JobDoneFile jobDoneFile = (JobDoneFile)this.objectMapper.readValue(new File(this.baseWorkingDir + "/" + id + "/" + "genie/genie.done"), JobDoneFile.class);
            File killReasonFile = new File(this.baseWorkingDir + "/" + id + "/" + "genie/kill-reason");
            String killedStatusMessages = killReasonFile.exists() ? ((JobKillReasonFile)this.objectMapper.readValue(killReasonFile, JobKillReasonFile.class)).getKillReason() : "Job was killed by user.";
            int exitCode = jobDoneFile.getExitCode();
            File stdOut = new File(jobDir, "stdout");
            Long stdOutSize = stdOut.exists() && stdOut.isFile() ? Long.valueOf(stdOut.length()) : null;
            File stdErr = new File(jobDir, "stderr");
            Long stdErrSize = stdErr.exists() && stdErr.isFile() ? Long.valueOf(stdErr.length()) : null;
            switch (exitCode) {
                case 999: {
                    this.jobPersistenceService.setJobCompletionInformation(id, exitCode, JobStatus.KILLED, killedStatusMessages, stdOutSize, stdErrSize);
                    finalStatus = JobStatus.KILLED;
                    break;
                }
                case 0: {
                    this.jobPersistenceService.setJobCompletionInformation(id, exitCode, JobStatus.SUCCEEDED, "Job finished successfully.", stdOutSize, stdErrSize);
                    finalStatus = JobStatus.SUCCEEDED;
                    break;
                }
                default: {
                    this.jobPersistenceService.setJobCompletionInformation(id, exitCode, JobStatus.FAILED, "Job failed.", stdOutSize, stdErrSize);
                    finalStatus = JobStatus.FAILED;
                }
            }
            return finalStatus;
        }
        catch (IOException ioe) {
            this.incrementErrorCounter("JOB_FINAL_UPDATE_FAILURE", ioe);
            log.error("Could not load the done file for job {}. Marking it as failed.", (Object)id, (Object)ioe);
            this.jobPersistenceService.updateJobStatus(id, JobStatus.FAILED, "Failed to load done file.");
            return JobStatus.FAILED;
        }
        catch (Throwable t) {
            this.incrementErrorCounter("JOB_FINAL_UPDATE_FAILURE", t);
            throw t;
        }
    }

    private void deleteApplicationDependencies(String jobId, File jobDir) {
        log.debug("Deleting dependencies as its enabled.");
        if (jobDir.exists()) {
            try {
                List appIds = this.jobSearchService.getJobApplications(jobId).stream().map(BaseDTO::getId).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toList());
                for (String appId : appIds) {
                    File appDependencyDir = new File(jobDir, "genie/applications/" + appId + "/" + "dependencies");
                    this.deleteDependenciesDirectory(appDependencyDir);
                }
            }
            catch (Exception e) {
                log.error("Could not delete job dependencies after completion for job: {} due to error {}", (Object)jobId, (Object)e);
                this.incrementErrorCounter("DELETE_APPLICATION_DEPENDENCIES_FAILURE", e);
            }
        }
    }

    private void deleteClusterDependencies(String jobId, File jobDir) {
        log.debug("Deleting dependencies as its enabled.");
        if (jobDir.exists()) {
            try {
                String clusterId = (String)this.jobSearchService.getJobCluster(jobId).getId().orElseThrow(IllegalStateException::new);
                File clusterDependencyDir = new File(jobDir, "genie/cluster/" + clusterId + "/" + "dependencies");
                this.deleteDependenciesDirectory(clusterDependencyDir);
            }
            catch (Exception e) {
                log.error("Could not delete job dependencies after completion for job: {} due to error {}", (Object)jobId, (Object)e);
                this.incrementErrorCounter("DELETE_CLUSTER_DEPENDENCIES_FAILURE", e);
            }
        }
    }

    private void deleteDependenciesDirectory(File dependencyDirectory) throws IOException {
        if (dependencyDirectory.exists()) {
            try {
                if (this.runAsUserEnabled) {
                    CommandLine deleteCommand = new CommandLine("sudo");
                    deleteCommand.addArgument("rm");
                    deleteCommand.addArgument("-rf");
                    deleteCommand.addArgument(dependencyDirectory.getCanonicalPath());
                    log.debug("Delete command is {}", (Object)deleteCommand);
                    this.executor.execute(deleteCommand);
                } else {
                    FileUtils.deleteDirectory((File)dependencyDirectory);
                }
            }
            catch (Throwable t) {
                this.incrementErrorCounter("DELETE_DEPENDENCIES_FAILURE");
                throw t;
            }
        }
    }

    private boolean processJobDir(Job job) throws GenieException, IOException {
        String jobId;
        File jobDir;
        log.debug("Got a job finished event. Will process job directory.");
        boolean result = false;
        Optional oJobId = job.getId();
        if (oJobId.isPresent() && !this.jobSearchService.getJobStatus((String)job.getId().get()).equals((Object)JobStatus.INVALID) && (jobDir = new File(this.baseWorkingDir, jobId = (String)oJobId.get())).exists()) {
            Optional archiveLocation;
            if (this.deleteDependencies) {
                this.deleteApplicationDependencies(jobId, jobDir);
                this.deleteClusterDependencies(jobId, jobDir);
            }
            if ((archiveLocation = job.getArchiveLocation()).isPresent() && !Strings.isNullOrEmpty((String)((String)archiveLocation.get()))) {
                CommandLine commandLine;
                log.debug("Archiving job directory");
                File localArchiveFile = new File(jobDir, "genie/logs/" + jobId + ".tar.gz");
                if (this.runAsUserEnabled) {
                    commandLine = new CommandLine("sudo");
                    commandLine.addArgument("tar");
                } else {
                    commandLine = new CommandLine("tar");
                }
                commandLine.addArgument("-c");
                commandLine.addArgument("-z");
                commandLine.addArgument("-f");
                commandLine.addArgument(localArchiveFile.getCanonicalPath());
                commandLine.addArgument("./");
                this.executor.setWorkingDirectory(jobDir);
                log.debug("Archive command : {}", (Object)commandLine);
                try {
                    this.executor.execute(commandLine);
                }
                catch (Throwable t) {
                    this.incrementErrorCounter("JOB_ARCHIVAL_FAILURE", t);
                    throw t;
                }
                this.genieFileTransferService.putFile(localArchiveFile.getCanonicalPath(), (String)archiveLocation.get());
                if (this.deleteArchiveFile) {
                    log.debug("Deleting archive file");
                    try {
                        if (this.runAsUserEnabled) {
                            CommandLine deleteCommand = new CommandLine("sudo");
                            deleteCommand.addArgument("rm");
                            deleteCommand.addArgument("-f");
                            deleteCommand.addArgument(localArchiveFile.getCanonicalPath());
                            this.executor.setWorkingDirectory(jobDir);
                            log.debug("Delete command: {}", (Object)deleteCommand);
                            this.executor.execute(deleteCommand);
                        } else if (!localArchiveFile.delete()) {
                            log.error("Failed to delete archive file for job: {}", (Object)jobId);
                            this.incrementErrorCounter("JOB_ARCHIVE_DELETION_FAILURE");
                        }
                    }
                    catch (Exception e) {
                        log.error("Failed to delete archive file for job: {}", (Object)jobId, (Object)e);
                        this.incrementErrorCounter("JOB_ARCHIVE_DELETION_FAILURE", e);
                    }
                }
                result = true;
            }
        }
        return result;
    }

    private boolean sendEmail(String jobId) throws GenieException {
        JobRequest jobRequest = this.jobSearchService.getJobRequest(jobId);
        boolean result = false;
        Optional email = jobRequest.getEmail();
        if (email.isPresent() && !Strings.isNullOrEmpty((String)((String)email.get()))) {
            log.debug("Got a job finished event. Sending email: {}", email.get());
            JobStatus status = this.jobSearchService.getJobStatus(jobId);
            StringBuilder subject = new StringBuilder().append("Genie Job Finished. Id: [").append(jobId).append("], Name: [").append(jobRequest.getName()).append("], Status: [").append(status).append("].");
            StringBuilder body = new StringBuilder().append("Id: [" + jobId + "]\n").append("Name: [" + jobRequest.getName() + "]\n").append("Status: [" + status + "]\n").append("User: [" + jobRequest.getUser() + "]\n").append("Tags: " + jobRequest.getTags() + "\n");
            jobRequest.getDescription().ifPresent(description -> body.append("[" + description + "]"));
            try {
                this.mailServiceImpl.sendEmail((String)email.get(), subject.toString(), body.toString());
            }
            catch (Throwable t) {
                this.incrementErrorCounter("JOB_EMAIL_FAILURE", t);
                throw t;
            }
            result = true;
        }
        return result;
    }

    private void incrementErrorCounter(String errorTagValue, Throwable throwable) {
        this.incrementErrorCounter((Map<String, String>)ImmutableMap.of((Object)ERROR_SOURCE_TAG, (Object)errorTagValue, (Object)"exceptionClass", (Object)throwable.getClass().getCanonicalName()));
    }

    private void incrementErrorCounter(String errorTagValue) {
        this.incrementErrorCounter((Map<String, String>)ImmutableMap.of((Object)ERROR_SOURCE_TAG, (Object)errorTagValue));
    }

    private void incrementErrorCounter(Map<String, String> tags) {
        this.registry.counter(this.errorCounterId.withTags(tags)).increment();
    }
}

