/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.genie.web.tasks.leader;

import com.google.common.collect.Sets;
import com.netflix.genie.common.internal.jobs.JobConstants;
import com.netflix.genie.web.properties.DatabaseCleanupProperties;
import com.netflix.genie.web.services.ClusterPersistenceService;
import com.netflix.genie.web.services.FilePersistenceService;
import com.netflix.genie.web.services.JobPersistenceService;
import com.netflix.genie.web.services.TagPersistenceService;
import com.netflix.genie.web.tasks.GenieTaskScheduleType;
import com.netflix.genie.web.tasks.TaskUtils;
import com.netflix.genie.web.tasks.leader.LeadershipTask;
import com.netflix.genie.web.util.MetricsUtils;
import io.micrometer.core.instrument.MeterRegistry;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.HashSet;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import javax.validation.constraints.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.scheduling.Trigger;
import org.springframework.scheduling.support.CronTrigger;

public class DatabaseCleanupTask
extends LeadershipTask {
    private static final Logger log = LoggerFactory.getLogger(DatabaseCleanupTask.class);
    private static final String DATABASE_CLEANUP_DURATION_TIMER_NAME = "genie.tasks.databaseCleanup.duration.timer";
    private final DatabaseCleanupProperties cleanupProperties;
    private final JobPersistenceService jobPersistenceService;
    private final ClusterPersistenceService clusterPersistenceService;
    private final FilePersistenceService filePersistenceService;
    private final TagPersistenceService tagPersistenceService;
    private final MeterRegistry registry;
    private final AtomicLong numDeletedJobs;
    private final AtomicLong numDeletedClusters;
    private final AtomicLong numDeletedTags;
    private final AtomicLong numDeletedFiles;

    public DatabaseCleanupTask(@NotNull DatabaseCleanupProperties cleanupProperties, @NotNull JobPersistenceService jobPersistenceService, @NotNull ClusterPersistenceService clusterPersistenceService, @NotNull FilePersistenceService filePersistenceService, @NotNull TagPersistenceService tagPersistenceService, @NotNull MeterRegistry registry) {
        this.registry = registry;
        this.cleanupProperties = cleanupProperties;
        this.jobPersistenceService = jobPersistenceService;
        this.clusterPersistenceService = clusterPersistenceService;
        this.filePersistenceService = filePersistenceService;
        this.tagPersistenceService = tagPersistenceService;
        this.numDeletedJobs = (AtomicLong)this.registry.gauge("genie.tasks.databaseCleanup.numDeletedJobs.gauge", (Number)new AtomicLong());
        this.numDeletedClusters = (AtomicLong)this.registry.gauge("genie.tasks.databaseCleanup.numDeletedClusters.gauge", (Number)new AtomicLong());
        this.numDeletedTags = (AtomicLong)this.registry.gauge("genie.tasks.databaseCleanup.numDeletedTags.gauge", (Number)new AtomicLong());
        this.numDeletedFiles = (AtomicLong)this.registry.gauge("genie.tasks.databaseCleanup.numDeletedFiles.gauge", (Number)new AtomicLong());
    }

    @Override
    public GenieTaskScheduleType getScheduleType() {
        return GenieTaskScheduleType.TRIGGER;
    }

    @Override
    public Trigger getTrigger() {
        return new CronTrigger(this.cleanupProperties.getExpression(), JobConstants.UTC);
    }

    @Override
    public void run() {
        long start = System.nanoTime();
        HashSet tags = Sets.newHashSet();
        try {
            if (this.cleanupProperties.isSkipJobsCleanup()) {
                log.debug("Skipping job cleanup");
                this.numDeletedJobs.set(0L);
            } else {
                long countDeletedJobs = this.deleteJobs();
                log.info("Deleted {} jobs", (Object)countDeletedJobs);
                this.numDeletedJobs.set(countDeletedJobs);
            }
            if (this.cleanupProperties.isSkipClustersCleanup()) {
                log.debug("Skipping clusters cleanup");
                this.numDeletedClusters.set(0L);
            } else {
                long countDeletedClusters = this.clusterPersistenceService.deleteTerminatedClusters();
                log.info("Deleted {} clusters that were in TERMINATED state and weren't attached to any jobs", (Object)countDeletedClusters);
                this.numDeletedClusters.set(countDeletedClusters);
            }
            Instant creationThreshold = Instant.now().minus(1L, ChronoUnit.HOURS);
            if (this.cleanupProperties.isSkipFilesCleanup()) {
                log.debug("Skipping files cleanup");
                this.numDeletedFiles.set(0L);
            } else {
                long countDeletedFiles = this.filePersistenceService.deleteUnusedFiles(creationThreshold);
                log.info("Deleted {} files that were unused by any resource and created over an hour ago", (Object)countDeletedFiles);
                this.numDeletedFiles.set(countDeletedFiles);
            }
            if (this.cleanupProperties.isSkipTagsCleanup()) {
                log.debug("Skipping tags cleanup");
                this.numDeletedTags.set(0L);
            } else {
                long countDeletedTags = this.tagPersistenceService.deleteUnusedTags(creationThreshold);
                log.info("Deleted {} tags that were unused by any resource and created over an hour ago", (Object)countDeletedTags);
                this.numDeletedTags.set(countDeletedTags);
            }
            MetricsUtils.addSuccessTags(tags);
        }
        catch (Throwable t) {
            MetricsUtils.addFailureTagsWithException(tags, t);
            throw t;
        }
        finally {
            this.registry.timer(DATABASE_CLEANUP_DURATION_TIMER_NAME, (Iterable)tags).record(System.nanoTime() - start, TimeUnit.NANOSECONDS);
        }
    }

    @Override
    public void cleanup() {
        this.numDeletedJobs.set(0L);
        this.numDeletedClusters.set(0L);
        this.numDeletedTags.set(0L);
        this.numDeletedFiles.set(0L);
    }

    private long deleteJobs() {
        long numberDeletedJobs;
        Instant midnightUTC = TaskUtils.getMidnightUTC();
        Instant retentionLimit = midnightUTC.minus(this.cleanupProperties.getRetention(), ChronoUnit.DAYS);
        int batchSize = this.cleanupProperties.getMaxDeletedPerTransaction();
        int pageSize = this.cleanupProperties.getPageSize();
        log.info("Attempting to delete jobs from before {} in batches of {} jobs per iteration", (Object)retentionLimit, (Object)batchSize);
        long totalDeletedJobs = 0L;
        do {
            numberDeletedJobs = this.jobPersistenceService.deleteBatchOfJobsCreatedBeforeDate(retentionLimit, batchSize, pageSize);
            totalDeletedJobs += numberDeletedJobs;
        } while (numberDeletedJobs != 0L);
        return totalDeletedJobs;
    }
}

