/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.genie.web.services.impl;

import com.amazonaws.SdkClientException;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3URI;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import com.netflix.genie.common.internal.aws.s3.S3ClientFactory;
import com.netflix.genie.web.exceptions.checked.AttachmentTooLargeException;
import com.netflix.genie.web.exceptions.checked.SaveAttachmentException;
import com.netflix.genie.web.properties.AttachmentServiceProperties;
import com.netflix.genie.web.services.AttachmentService;
import com.netflix.genie.web.util.MetricsUtils;
import io.micrometer.core.instrument.MeterRegistry;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashSet;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;

public class S3AttachmentServiceImpl
implements AttachmentService {
    private static final Logger log = LoggerFactory.getLogger(S3AttachmentServiceImpl.class);
    private static final String METRICS_PREFIX = "genie.jobs.attachments.s3";
    private static final String COUNT_DISTRIBUTION = "genie.jobs.attachments.s3.count.distribution";
    private static final String LARGEST_SIZE_DISTRIBUTION = "genie.jobs.attachments.s3.largest.distribution";
    private static final String TOTAL_SIZE_DISTRIBUTION = "genie.jobs.attachments.s3.totalSize.distribution";
    private static final String SAVE_TIMER = "genie.jobs.attachments.s3.upload.timer";
    private static final Set<URI> EMPTY_SET = ImmutableSet.of();
    private static final String SLASH = "/";
    private static final String S3 = "s3";
    private final S3ClientFactory s3ClientFactory;
    private final AttachmentServiceProperties properties;
    private final MeterRegistry meterRegistry;
    private final AmazonS3URI s3BaseURI;

    public S3AttachmentServiceImpl(S3ClientFactory s3ClientFactory, AttachmentServiceProperties attachmentServiceProperties, MeterRegistry meterRegistry) {
        this.s3ClientFactory = s3ClientFactory;
        this.properties = attachmentServiceProperties;
        this.meterRegistry = meterRegistry;
        this.s3BaseURI = new AmazonS3URI(attachmentServiceProperties.getLocationPrefix());
    }

    @Override
    public Set<URI> saveAttachments(@Nullable String jobId, Set<Resource> attachments) throws SaveAttachmentException {
        this.meterRegistry.summary(COUNT_DISTRIBUTION, new String[0]).record((double)attachments.size());
        log.debug("Saving {} attachments for job request with id: {}", (Object)attachments.size(), (Object)jobId);
        if (attachments.size() == 0) {
            return EMPTY_SET;
        }
        this.checkLimits(attachments);
        long start = System.nanoTime();
        HashSet tags = Sets.newHashSet();
        try {
            Set<URI> attachmentURIs = this.uploadAllAttachments(jobId, attachments);
            MetricsUtils.addSuccessTags(tags);
            Set<URI> set = attachmentURIs;
            return set;
        }
        catch (SaveAttachmentException e) {
            log.error("Failed to save attachments (requested job id: {}): {}", new Object[]{jobId, e.getMessage(), e});
            MetricsUtils.addFailureTagsWithException(tags, (Throwable)((Object)e));
            throw e;
        }
        finally {
            this.meterRegistry.timer(SAVE_TIMER, (Iterable)tags).record(System.nanoTime() - start, TimeUnit.NANOSECONDS);
        }
    }

    private void checkLimits(Set<Resource> attachments) throws SaveAttachmentException {
        long singleSizeLimit = this.properties.getMaxSize().toBytes();
        long totalSizeLimit = this.properties.getMaxTotalSize().toBytes();
        long totalSize = 0L;
        long largestSize = 0L;
        for (Resource attachment : attachments) {
            long attachmentSize;
            String filename = attachment.getFilename();
            try {
                attachmentSize = attachment.contentLength();
            }
            catch (IOException e) {
                throw new SaveAttachmentException("Failed to get size of attachment: " + filename + ": " + e.getMessage(), e);
            }
            if (attachmentSize > largestSize) {
                largestSize = attachmentSize;
            }
            totalSize += attachmentSize;
        }
        if (largestSize > singleSizeLimit) {
            throw new AttachmentTooLargeException("Size of attachment exceeds the maximum allowed (" + largestSize + " > " + singleSizeLimit + ")");
        }
        if (totalSize > totalSizeLimit) {
            throw new AttachmentTooLargeException("Total size of attachments exceeds the maximum allowed (" + totalSize + " > " + totalSizeLimit + ")");
        }
        this.meterRegistry.summary(LARGEST_SIZE_DISTRIBUTION, new String[0]).record((double)largestSize);
        this.meterRegistry.summary(TOTAL_SIZE_DISTRIBUTION, new String[0]).record((double)totalSize);
    }

    private Set<URI> uploadAllAttachments(@Nullable String jobId, Set<Resource> attachments) throws SaveAttachmentException {
        AmazonS3 s3Client = this.s3ClientFactory.getClient(this.s3BaseURI);
        String bundleId = UUID.randomUUID().toString();
        String commonPrefix = this.s3BaseURI.getKey() + SLASH + bundleId + SLASH;
        log.debug("Uploading {} attachments for job request with id {} to: {}", new Object[]{attachments.size(), jobId, commonPrefix});
        HashSet attachmentURIs = Sets.newHashSet();
        for (Resource attachment : attachments) {
            String filename = attachment.getFilename();
            if (StringUtils.isBlank((CharSequence)filename)) {
                throw new SaveAttachmentException("Attachment filename is missing");
            }
            String objectBucket = this.s3BaseURI.getBucket();
            String objectKey = commonPrefix + filename;
            ObjectMetadata metadata = new ObjectMetadata();
            URI attachmentURI = null;
            try {
                InputStream inputStream = attachment.getInputStream();
                Throwable throwable = null;
                try {
                    metadata.setContentLength(attachment.contentLength());
                    attachmentURI = new URI(S3, objectBucket, SLASH + objectKey, null);
                    s3Client.putObject(objectBucket, objectKey, inputStream, metadata);
                    attachmentURIs.add(attachmentURI);
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (inputStream == null) continue;
                    if (throwable != null) {
                        try {
                            inputStream.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    inputStream.close();
                }
            }
            catch (SdkClientException | IOException | URISyntaxException e) {
                throw new SaveAttachmentException("Failed to upload attachment: " + attachmentURI + " - " + e.getMessage(), e);
            }
        }
        return attachmentURIs;
    }
}

