/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.genie.web.services.impl;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.netflix.genie.common.exceptions.GenieException;
import com.netflix.genie.common.exceptions.GenieNotFoundException;
import com.netflix.genie.common.exceptions.GeniePreconditionException;
import com.netflix.genie.common.exceptions.GenieServerException;
import com.netflix.genie.common.exceptions.GenieServerUnavailableException;
import com.netflix.genie.common.external.dtos.v4.ArchiveStatus;
import com.netflix.genie.common.external.util.GenieObjectMapper;
import com.netflix.genie.common.internal.dtos.DirectoryManifest;
import com.netflix.genie.web.agent.resources.AgentFileProtocolResolver;
import com.netflix.genie.web.agent.services.AgentFileStreamService;
import com.netflix.genie.web.agent.services.AgentRoutingService;
import com.netflix.genie.web.data.services.DataServices;
import com.netflix.genie.web.data.services.PersistenceService;
import com.netflix.genie.web.dtos.ArchivedJobMetadata;
import com.netflix.genie.web.exceptions.checked.JobDirectoryManifestNotFoundException;
import com.netflix.genie.web.exceptions.checked.JobNotArchivedException;
import com.netflix.genie.web.exceptions.checked.JobNotFoundException;
import com.netflix.genie.web.exceptions.checked.NotFoundException;
import com.netflix.genie.web.resources.writers.DefaultDirectoryWriter;
import com.netflix.genie.web.services.ArchivedJobService;
import com.netflix.genie.web.services.JobDirectoryServerService;
import com.netflix.genie.web.util.MetricsUtils;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Tag;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.concurrent.TimeUnit;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.http.client.utils.URIBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.http.MediaType;
import org.springframework.http.converter.ResourceHttpMessageConverter;
import org.springframework.http.converter.ResourceRegionHttpMessageConverter;
import org.springframework.web.servlet.resource.ResourceHttpRequestHandler;

public class JobDirectoryServerServiceImpl
implements JobDirectoryServerService {
    private static final Logger log = LoggerFactory.getLogger(JobDirectoryServerServiceImpl.class);
    private static final String SLASH = "/";
    private static final String SERVE_RESOURCE_TIMER = "genie.files.serve.timer";
    private static final String ARCHIVE_STATUS_TAG = "archiveStatus";
    private final ResourceLoader resourceLoader;
    private final PersistenceService persistenceService;
    private final AgentFileStreamService agentFileStreamService;
    private final MeterRegistry meterRegistry;
    private final GenieResourceHandler.Factory genieResourceHandlerFactory;
    private final ArchivedJobService archivedJobService;
    private final AgentRoutingService agentRoutingService;

    public JobDirectoryServerServiceImpl(ResourceLoader resourceLoader, DataServices dataServices, AgentFileStreamService agentFileStreamService, ArchivedJobService archivedJobService, MeterRegistry meterRegistry, AgentRoutingService agentRoutingService) {
        this(resourceLoader, dataServices, agentFileStreamService, archivedJobService, new GenieResourceHandler.Factory(), meterRegistry, agentRoutingService);
    }

    @VisibleForTesting
    JobDirectoryServerServiceImpl(ResourceLoader resourceLoader, DataServices dataServices, AgentFileStreamService agentFileStreamService, ArchivedJobService archivedJobService, GenieResourceHandler.Factory genieResourceHandlerFactory, MeterRegistry meterRegistry, AgentRoutingService agentRoutingService) {
        this.resourceLoader = resourceLoader;
        this.persistenceService = dataServices.getPersistenceService();
        this.agentFileStreamService = agentFileStreamService;
        this.meterRegistry = meterRegistry;
        this.genieResourceHandlerFactory = genieResourceHandlerFactory;
        this.archivedJobService = archivedJobService;
        this.agentRoutingService = agentRoutingService;
    }

    @Override
    public void serveResource(String id, URL baseUrl, String relativePath, HttpServletRequest request, HttpServletResponse response) throws GenieException {
        long start = System.nanoTime();
        HashSet tags = Sets.newHashSet();
        try {
            URI jobDirRoot;
            DirectoryManifest manifest;
            URI baseUri = new URI(baseUrl.toString() + SLASH).normalize();
            ArchiveStatus archiveStatus = this.persistenceService.getJobArchiveStatus(id);
            tags.add(Tag.of((String)ARCHIVE_STATUS_TAG, (String)archiveStatus.name()));
            switch (archiveStatus) {
                case NO_FILES: {
                    throw new GenieNotFoundException("Job failed before any file was created: " + id);
                }
                case FAILED: {
                    throw new GenieNotFoundException("Job failed to archive files: " + id);
                }
                case DISABLED: {
                    throw new GeniePreconditionException("Archive disabled for job " + id);
                }
                case UNKNOWN: 
                case ARCHIVED: {
                    log.debug("Routing request to archive");
                    ArchivedJobMetadata archivedJobMetadata = this.archivedJobService.getArchivedJobMetadata(id);
                    String rangeHeader = request.getHeader("Range");
                    manifest = archivedJobMetadata.getManifest();
                    URI baseJobDirRoot = archivedJobMetadata.getArchiveBaseUri();
                    jobDirRoot = new URIBuilder(baseJobDirRoot).setFragment(rangeHeader).build();
                    break;
                }
                case PENDING: {
                    log.debug("Routing request to connected agent");
                    if (!this.agentRoutingService.isAgentConnectionLocal(id)) {
                        throw new GenieServerUnavailableException("Agent connection has moved or was terminated");
                    }
                    manifest = this.agentFileStreamService.getManifest(id).orElseThrow(() -> new GenieServerUnavailableException("Manifest not found for job " + id));
                    jobDirRoot = AgentFileProtocolResolver.createUri(id, SLASH, request.getHeader("Range"));
                    break;
                }
                default: {
                    throw new GenieServerException("Unknown archive status " + archiveStatus + "(" + id + ")");
                }
            }
            log.debug("Serving file: {} for job: {} (archive status: {})", new Object[]{relativePath, id, archiveStatus});
            this.handleRequest(baseUri, relativePath, request, response, manifest, jobDirRoot);
            MetricsUtils.addSuccessTags(tags);
        }
        catch (NotFoundException e) {
            MetricsUtils.addFailureTagsWithException(tags, (Throwable)((Object)e));
            throw new GenieNotFoundException(e.getMessage(), (Throwable)((Object)e));
        }
        catch (IOException e) {
            MetricsUtils.addFailureTagsWithException(tags, e);
            throw new GenieServerException("Error serving response: " + e.getMessage(), (Throwable)e);
        }
        catch (URISyntaxException e) {
            MetricsUtils.addFailureTagsWithException(tags, e);
            throw new GenieServerException(e.getMessage(), (Throwable)e);
        }
        catch (JobNotArchivedException e) {
            MetricsUtils.addFailureTagsWithException(tags, (Throwable)((Object)e));
            throw new GeniePreconditionException("Job outputs were not archived", (Throwable)((Object)e));
        }
        catch (JobDirectoryManifestNotFoundException | JobNotFoundException e) {
            MetricsUtils.addFailureTagsWithException(tags, e);
            throw new GenieNotFoundException("Failed to retrieve job archived files metadata", (Throwable)e);
        }
        catch (GenieException e) {
            MetricsUtils.addFailureTagsWithException(tags, e);
            throw e;
        }
        finally {
            long elapsed = System.nanoTime() - start;
            this.meterRegistry.timer(SERVE_RESOURCE_TIMER, (Iterable)tags).record(elapsed, TimeUnit.NANOSECONDS);
        }
    }

    private void handleRequest(URI baseUri, String relativePath, HttpServletRequest request, HttpServletResponse response, DirectoryManifest manifest, URI jobDirectoryRoot) throws IOException, GenieNotFoundException, GenieServerException {
        log.debug("Handle request, baseUri: '{}', relpath: '{}', jobRootUri: '{}'", new Object[]{baseUri, relativePath, jobDirectoryRoot});
        DirectoryManifest.ManifestEntry entry = (DirectoryManifest.ManifestEntry)manifest.getEntry(relativePath).orElseThrow(() -> new GenieNotFoundException("No such entry in job manifest: " + relativePath));
        if (entry.isDirectory()) {
            DefaultDirectoryWriter.Directory directory = new DefaultDirectoryWriter.Directory();
            ArrayList files = Lists.newArrayList();
            ArrayList directories = Lists.newArrayList();
            try {
                entry.getParent().ifPresent(parentPath -> {
                    DirectoryManifest.ManifestEntry parentEntry = (DirectoryManifest.ManifestEntry)manifest.getEntry(parentPath).orElseThrow(IllegalArgumentException::new);
                    directory.setParent(this.createEntry(parentEntry, baseUri));
                });
                for (String childPath : entry.getChildren()) {
                    DirectoryManifest.ManifestEntry childEntry = (DirectoryManifest.ManifestEntry)manifest.getEntry(childPath).orElseThrow(IllegalArgumentException::new);
                    if (childEntry.isDirectory()) {
                        directories.add(this.createEntry(childEntry, baseUri));
                        continue;
                    }
                    files.add(this.createEntry(childEntry, baseUri));
                }
            }
            catch (IllegalArgumentException iae) {
                throw new GenieServerException("Error while traversing files manifest: " + iae.getMessage(), (Throwable)iae);
            }
            directories.sort(Comparator.comparing(DefaultDirectoryWriter.Entry::getName));
            files.sort(Comparator.comparing(DefaultDirectoryWriter.Entry::getName));
            directory.setDirectories(directories);
            directory.setFiles(files);
            String accept = request.getHeader("Accept");
            if (accept != null && accept.contains("text/html")) {
                response.setContentType("text/html");
                response.getOutputStream().write(DefaultDirectoryWriter.directoryToHTML(entry.getName(), directory).getBytes(StandardCharsets.UTF_8));
            } else {
                response.setContentType("application/json");
                GenieObjectMapper.getMapper().writeValue((OutputStream)response.getOutputStream(), (Object)directory);
            }
        } else {
            URI location = jobDirectoryRoot.resolve(entry.getPath());
            String locationString = location.toString() + (jobDirectoryRoot.getFragment() != null ? "#" + jobDirectoryRoot.getFragment() : "");
            log.debug("Get resource: {}", (Object)locationString);
            Resource jobResource = this.resourceLoader.getResource(locationString);
            String mediaType = entry.getMimeType().orElse("text/plain");
            ResourceHttpRequestHandler handler = this.genieResourceHandlerFactory.get(mediaType, jobResource);
            try {
                handler.handleRequest(request, response);
            }
            catch (ServletException e) {
                throw new GenieServerException("Servlet exception: " + e.getMessage(), (Throwable)e);
            }
        }
    }

    private DefaultDirectoryWriter.Entry createEntry(DirectoryManifest.ManifestEntry manifestEntry, URI baseUri) {
        DefaultDirectoryWriter.Entry entry = new DefaultDirectoryWriter.Entry();
        if (manifestEntry.isDirectory()) {
            entry.setName(manifestEntry.getName().endsWith(SLASH) ? manifestEntry.getName() : manifestEntry.getName() + SLASH);
        } else {
            entry.setName(manifestEntry.getName());
        }
        entry.setUrl(baseUri.resolve(manifestEntry.getPath()).toString());
        entry.setSize(manifestEntry.getSize());
        entry.setLastModified(manifestEntry.getLastModifiedTime());
        return entry;
    }

    private static class GenieResourceHandler
    extends ResourceHttpRequestHandler {
        private static final ResourceHttpMessageConverter RESOURCE_HTTP_MESSAGE_CONVERTER = new ResourceHttpMessageConverter();
        private static final ResourceRegionHttpMessageConverter RESOURCE_REGION_HTTP_MESSAGE_CONVERTER = new ResourceRegionHttpMessageConverter();
        private final MediaType mediaType;
        private final Resource jobResource;

        GenieResourceHandler(String mediaType, Resource jobResource) {
            this.mediaType = MediaType.parseMediaType((String)mediaType);
            this.jobResource = jobResource;
            this.setResourceHttpMessageConverter(RESOURCE_HTTP_MESSAGE_CONVERTER);
            this.setResourceRegionHttpMessageConverter(RESOURCE_REGION_HTTP_MESSAGE_CONVERTER);
        }

        protected Resource getResource(HttpServletRequest request) throws IOException {
            return this.jobResource;
        }

        protected MediaType getMediaType(HttpServletRequest request, Resource resource) {
            return this.mediaType;
        }

        private static class Factory {
            private Factory() {
            }

            ResourceHttpRequestHandler get(String mediaType, Resource jobResource) {
                return new GenieResourceHandler(mediaType, jobResource);
            }
        }
    }
}

