/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.genie.web.agent.services.impl;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.netflix.genie.web.agent.services.AgentConnectionTrackingService;
import com.netflix.genie.web.agent.services.AgentRoutingService;
import com.netflix.genie.web.properties.AgentConnectionTrackingServiceProperties;
import java.time.Instant;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.actuate.info.Info;
import org.springframework.boot.actuate.info.InfoContributor;
import org.springframework.scheduling.TaskScheduler;

public class AgentConnectionTrackingServiceImpl
implements AgentConnectionTrackingService,
InfoContributor {
    private static final Logger log = LoggerFactory.getLogger(AgentConnectionTrackingServiceImpl.class);
    private final AgentRoutingService agentRoutingService;
    private final TaskScheduler taskScheduler;
    private final HashMap<String, JobStreamsRecord> jobStreamRecordsMap = Maps.newHashMap();
    private final AgentConnectionTrackingServiceProperties serviceProperties;
    private final Supplier<Instant> timeSupplier;

    public AgentConnectionTrackingServiceImpl(AgentRoutingService agentRoutingService, TaskScheduler taskScheduler, AgentConnectionTrackingServiceProperties serviceProperties) {
        this(agentRoutingService, taskScheduler, serviceProperties, Instant::now);
    }

    @VisibleForTesting
    AgentConnectionTrackingServiceImpl(AgentRoutingService agentRoutingService, TaskScheduler taskScheduler, AgentConnectionTrackingServiceProperties serviceProperties, Supplier<Instant> timeSupplier) {
        this.agentRoutingService = agentRoutingService;
        this.taskScheduler = taskScheduler;
        this.serviceProperties = serviceProperties;
        this.timeSupplier = timeSupplier;
        this.taskScheduler.scheduleAtFixedRate(this::cleanupTask, this.serviceProperties.getCleanupInterval());
    }

    @Override
    public synchronized void notifyHeartbeat(String streamId, String claimedJobId) {
        JobStreamsRecord record;
        boolean isNew = false;
        if (this.jobStreamRecordsMap.containsKey(claimedJobId)) {
            record = this.jobStreamRecordsMap.get(claimedJobId);
        } else {
            record = new JobStreamsRecord(claimedJobId);
            this.jobStreamRecordsMap.put(claimedJobId, record);
            isNew = true;
        }
        record.updateActiveStream(streamId, this.timeSupplier.get());
        log.debug("Received heartbeat for {} job {} using stream {}", new Object[]{isNew ? "new" : "existing", claimedJobId, streamId});
        if (isNew) {
            log.debug("Notify new agent connection for job {}", (Object)claimedJobId);
            this.agentRoutingService.handleClientConnected(claimedJobId);
        }
    }

    @Override
    public synchronized void notifyDisconnected(String streamId, String claimedJobId) {
        JobStreamsRecord jobStreamsRecord = this.jobStreamRecordsMap.get(claimedJobId);
        log.debug("Received disconnection for {} job {} using stream {}", new Object[]{jobStreamsRecord == null ? "unknown" : "existing", claimedJobId, streamId});
        if (jobStreamsRecord != null) {
            jobStreamsRecord.removeActiveStream(streamId);
            if (!jobStreamsRecord.hasActiveStreams()) {
                log.debug("Job {} last stream disconnected, notifying routing service", (Object)claimedJobId);
                this.jobStreamRecordsMap.remove(claimedJobId);
                this.agentRoutingService.handleClientDisconnected(claimedJobId);
            }
        }
    }

    @Override
    public synchronized long getConnectedAgentsCount() {
        return this.jobStreamRecordsMap.size();
    }

    private synchronized void cleanupTask() {
        Instant cutoff = this.timeSupplier.get().minus(this.serviceProperties.getConnectionExpirationPeriod());
        this.jobStreamRecordsMap.forEach((jobId, record) -> record.expungeExpiredStreams(cutoff));
        HashSet removedJobIds = Sets.newHashSet();
        this.jobStreamRecordsMap.entrySet().removeIf(entry -> {
            if (!((JobStreamsRecord)entry.getValue()).hasActiveStreams()) {
                removedJobIds.add((String)entry.getKey());
                return true;
            }
            return false;
        });
        for (String jobId2 : removedJobIds) {
            log.debug("Job {} last stream expired, notifying routing service", (Object)jobId2);
            this.agentRoutingService.handleClientDisconnected(jobId2);
        }
    }

    public void contribute(Info.Builder builder) {
        List<String> jobIds = this.getConnectedAgentsIds();
        builder.withDetail("connectedAgents", jobIds);
    }

    private synchronized List<String> getConnectedAgentsIds() {
        return ImmutableList.copyOf(this.jobStreamRecordsMap.keySet());
    }

    private static final class JobStreamsRecord {
        private final String jobId;
        private final Map<String, Instant> streamsLastHeartbeatMap = Maps.newHashMap();

        private JobStreamsRecord(String jobId) {
            this.jobId = jobId;
        }

        private void updateActiveStream(String streamId, Instant currentTime) {
            Instant previousHeartbeat = this.streamsLastHeartbeatMap.put(streamId, currentTime);
            log.debug("{} heartbeat for job {} stream {}", new Object[]{previousHeartbeat == null ? "Created" : "Updated", this.jobId, streamId});
        }

        private void removeActiveStream(String streamId) {
            Instant previousHeartbeat = this.streamsLastHeartbeatMap.remove(streamId);
            if (previousHeartbeat != null) {
                log.debug("Removed job {} stream {}", (Object)this.jobId, (Object)streamId);
            }
        }

        private boolean hasActiveStreams() {
            return !this.streamsLastHeartbeatMap.isEmpty();
        }

        private void expungeExpiredStreams(Instant cutoff) {
            boolean removed = this.streamsLastHeartbeatMap.entrySet().removeIf(entry -> ((Instant)entry.getValue()).isBefore(cutoff));
            if (removed) {
                log.debug("Removed expired streams for job {}", (Object)this.jobId);
            }
        }
    }
}

