/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.genie.web.services.impl;

import brave.SpanCustomizer;
import brave.Tracer;
import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.collect.Sets;
import com.netflix.genie.common.internal.dtos.ArchiveStatus;
import com.netflix.genie.common.internal.dtos.JobStatus;
import com.netflix.genie.common.internal.exceptions.checked.GenieJobResolutionException;
import com.netflix.genie.common.internal.tracing.brave.BraveTracingComponents;
import com.netflix.genie.web.agent.launchers.AgentLauncher;
import com.netflix.genie.web.data.services.DataServices;
import com.netflix.genie.web.data.services.PersistenceService;
import com.netflix.genie.web.dtos.JobSubmission;
import com.netflix.genie.web.dtos.ResolvedJob;
import com.netflix.genie.web.dtos.ResourceSelectionResult;
import com.netflix.genie.web.exceptions.checked.AgentLaunchException;
import com.netflix.genie.web.exceptions.checked.IdAlreadyExistsException;
import com.netflix.genie.web.exceptions.checked.NotFoundException;
import com.netflix.genie.web.exceptions.checked.ResourceSelectionException;
import com.netflix.genie.web.selectors.AgentLauncherSelectionContext;
import com.netflix.genie.web.selectors.AgentLauncherSelector;
import com.netflix.genie.web.services.JobLaunchService;
import com.netflix.genie.web.services.JobResolverService;
import com.netflix.genie.web.util.MetricsUtils;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Tag;
import java.util.Collection;
import java.util.HashSet;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nonnull;
import javax.validation.Valid;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JobLaunchServiceImpl
implements JobLaunchService {
    private static final Logger log = LoggerFactory.getLogger(JobLaunchServiceImpl.class);
    static final String BEGIN_LAUNCH_JOB_ANNOTATION = "Beginning to Launch Job";
    static final String SAVED_JOB_SUBMISSION_ANNOTATION = "Saved Job Submission";
    static final String RESOLVED_JOB_ANNOTATION = "Resolved Job";
    static final String MARKED_JOB_ACCEPTED_ANNOTATION = "Marked Job Accepted";
    static final String LAUNCHED_AGENT_ANNOTATION = "Launched Agent";
    static final String SAVED_LAUNCHER_EXT_ANNOTATION = "Saved Launcher Ext Data";
    static final String END_LAUNCH_JOB_ANNOTATION = "Completed Launching Job";
    private static final String LAUNCH_JOB_TIMER = "genie.services.jobLaunch.launchJob.timer";
    private static final String AGENT_LAUNCHER_SELECTOR_TIMER = "genie.services.jobLaunch.selectLauncher.timer";
    private static final String AVAILABLE_LAUNCHERS_TAG = "numAvailableLaunchers";
    private static final String SELECTOR_CLASS_TAG = "agentLauncherSelectorClass";
    private static final String LAUNCHER_CLASS_TAG = "agentLauncherSelectedClass";
    private static final int MAX_STATUS_UPDATE_ATTEMPTS = 5;
    private static final int INITIAL_ATTEMPT = 0;
    private static final String ACCEPTED_MESSAGE = "The job has been accepted by the system for execution";
    private final PersistenceService persistenceService;
    private final JobResolverService jobResolverService;
    private final AgentLauncherSelector agentLauncherSelector;
    private final Tracer tracer;
    private final MeterRegistry registry;

    public JobLaunchServiceImpl(DataServices dataServices, JobResolverService jobResolverService, AgentLauncherSelector agentLauncherSelector, BraveTracingComponents tracingComponents, MeterRegistry registry) {
        this.persistenceService = dataServices.getPersistenceService();
        this.jobResolverService = jobResolverService;
        this.agentLauncherSelector = agentLauncherSelector;
        this.tracer = tracingComponents.getTracer();
        this.registry = registry;
    }

    @Override
    @Nonnull
    public String launchJob(@Valid JobSubmission jobSubmission) throws AgentLaunchException, GenieJobResolutionException, IdAlreadyExistsException, NotFoundException {
        long start = System.nanoTime();
        SpanCustomizer span = this.tracer.currentSpanCustomizer();
        span.annotate(BEGIN_LAUNCH_JOB_ANNOTATION);
        HashSet tags = Sets.newHashSet();
        try {
            Optional<JsonNode> launcherExt;
            ResolvedJob resolvedJob;
            String jobId = this.persistenceService.saveJobSubmission(jobSubmission);
            span.annotate(SAVED_JOB_SUBMISSION_ANNOTATION);
            try {
                resolvedJob = this.jobResolverService.resolveJob(jobId);
            }
            catch (Throwable t) {
                String message = t instanceof GenieJobResolutionException ? "Failed to resolve job given original request and available resources" : "Runtime error during job resolution";
                MetricsUtils.addFailureTagsWithException(tags, t);
                this.persistenceService.updateJobArchiveStatus(jobId, ArchiveStatus.NO_FILES);
                if (this.updateJobStatus(jobId, JobStatus.RESERVED, JobStatus.FAILED, message, 0) != JobStatus.FAILED) {
                    log.error("Updating status to failed didn't succeed");
                }
                throw t;
            }
            span.annotate(RESOLVED_JOB_ANNOTATION);
            try {
                JobStatus updatedStatus = this.updateJobStatus(jobId, JobStatus.RESOLVED, JobStatus.ACCEPTED, ACCEPTED_MESSAGE, 0);
                if (updatedStatus != JobStatus.ACCEPTED) {
                    throw new AgentLaunchException("Unable to mark job accepted. Job state " + updatedStatus);
                }
            }
            catch (Exception e) {
                this.persistenceService.updateJobArchiveStatus(jobId, ArchiveStatus.NO_FILES);
                throw e;
            }
            span.annotate(MARKED_JOB_ACCEPTED_ANNOTATION);
            JsonNode requestedLauncherExt = this.persistenceService.getRequestedLauncherExt(jobId);
            try {
                AgentLauncher launcher = this.selectLauncher(jobId, jobSubmission, resolvedJob);
                tags.add(Tag.of((String)LAUNCHER_CLASS_TAG, (String)launcher.getClass().getCanonicalName()));
                launcherExt = launcher.launchAgent(resolvedJob, requestedLauncherExt);
            }
            catch (AgentLaunchException e) {
                this.persistenceService.updateJobArchiveStatus(jobId, ArchiveStatus.NO_FILES);
                this.updateJobStatus(jobId, JobStatus.ACCEPTED, JobStatus.FAILED, e.getMessage(), 0);
                throw e;
            }
            span.annotate(LAUNCHED_AGENT_ANNOTATION);
            if (launcherExt.isPresent()) {
                try {
                    this.persistenceService.updateLauncherExt(jobId, launcherExt.get());
                }
                catch (Exception e) {
                    log.error("Unable to update the launcher ext for job {}", (Object)jobId, (Object)e);
                }
            }
            span.annotate(SAVED_LAUNCHER_EXT_ANNOTATION);
            MetricsUtils.addSuccessTags(tags);
            String string = jobId;
            return string;
        }
        catch (Throwable t) {
            MetricsUtils.addFailureTagsWithException(tags, t);
            throw t;
        }
        finally {
            span.annotate(END_LAUNCH_JOB_ANNOTATION);
            this.registry.timer(LAUNCH_JOB_TIMER, (Iterable)tags).record(System.nanoTime() - start, TimeUnit.NANOSECONDS);
        }
    }

    private AgentLauncher selectLauncher(String jobId, JobSubmission jobSubmission, ResolvedJob resolvedJob) throws AgentLaunchException {
        Collection<AgentLauncher> availableLaunchers = this.agentLauncherSelector.getAgentLaunchers();
        log.debug("Selecting agent launcher for job {} ({} available)", (Object)jobId, (Object)availableLaunchers.size());
        AgentLauncherSelectionContext context = new AgentLauncherSelectionContext(jobId, jobSubmission.getJobRequest(), jobSubmission.getJobRequestMetadata(), resolvedJob, availableLaunchers);
        HashSet tags = Sets.newHashSet();
        long start = System.nanoTime();
        tags.add(Tag.of((String)AVAILABLE_LAUNCHERS_TAG, (String)String.valueOf(availableLaunchers.size())));
        tags.add(Tag.of((String)SELECTOR_CLASS_TAG, (String)this.agentLauncherSelector.getClass().getSimpleName()));
        try {
            ResourceSelectionResult selectionResult = this.agentLauncherSelector.select(context);
            AgentLauncher selectedLauncher = (AgentLauncher)selectionResult.getSelectedResource().orElseThrow(() -> new ResourceSelectionException("No AgentLauncher selected: " + selectionResult.getSelectionRationale().orElse("Rationale unknown")));
            MetricsUtils.addSuccessTags(tags);
            tags.add(Tag.of((String)LAUNCHER_CLASS_TAG, (String)selectedLauncher.getClass().getSimpleName()));
            log.debug("Selected launcher {} for job {}", (Object)selectedLauncher, (Object)jobId);
            AgentLauncher agentLauncher = selectedLauncher;
            return agentLauncher;
        }
        catch (ResourceSelectionException e) {
            log.error("Error selecting agent launcher", (Throwable)((Object)e));
            MetricsUtils.addFailureTagsWithException(tags, (Throwable)((Object)e));
            throw new AgentLaunchException("Failed to select an Agent Launcher", (Throwable)((Object)e));
        }
        finally {
            this.registry.timer(AGENT_LAUNCHER_SELECTOR_TIMER, (Iterable)tags).record(System.nanoTime() - start, TimeUnit.NANOSECONDS);
        }
    }

    private JobStatus updateJobStatus(String jobId, JobStatus expectedStatus, JobStatus desiredStatus, String desiredStatusMessage, int attemptNumber) throws NotFoundException {
        int nextAttemptNumber = attemptNumber + 1;
        JobStatus currentStatus = this.persistenceService.updateJobStatus(jobId, expectedStatus, desiredStatus, desiredStatusMessage);
        if (currentStatus.isFinished()) {
            log.info("Won't change job status of {} from {} to {} desired status as {} is already a final status", new Object[]{jobId, currentStatus, desiredStatus, currentStatus});
            return currentStatus;
        }
        if (currentStatus == desiredStatus) {
            log.debug("Successfully updated status of {} from {} to {}", new Object[]{jobId, expectedStatus, desiredStatus});
            return currentStatus;
        }
        log.error("Job {} status changed from expected {} to {}. Couldn't update to {}. Attempt {}", new Object[]{jobId, expectedStatus, currentStatus, desiredStatus, nextAttemptNumber});
        if (nextAttemptNumber < 5) {
            return this.updateJobStatus(jobId, currentStatus, desiredStatus, desiredStatusMessage, nextAttemptNumber);
        }
        log.error("Out of attempts to update job {} status to {}. Unable to complete status update", (Object)jobId, (Object)desiredStatus);
        return currentStatus;
    }
}

