/*
 * Copyright 2015-2016 Netflix, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.netflix.nebula.lint.plugin

import com.netflix.nebula.lint.Styling
import org.codenarc.AnalysisContext
import org.codenarc.report.HtmlReportWriter
import org.codenarc.report.ReportWriter
import org.codenarc.report.TextReportWriter
import org.codenarc.report.XmlReportWriter
import org.gradle.api.DefaultTask
import org.gradle.api.plugins.quality.CodeNarcReports
import org.gradle.api.plugins.quality.internal.CodeNarcReportsImpl
import org.gradle.api.reporting.Report
import org.gradle.api.reporting.Reporting
import org.gradle.api.tasks.Nested
import org.gradle.api.tasks.TaskAction
import org.gradle.api.tasks.VerificationTask
import org.gradle.internal.reflect.Instantiator
import org.gradle.logging.StyledTextOutputFactory

import javax.inject.Inject

class GradleLintReportTask extends DefaultTask implements VerificationTask, Reporting<CodeNarcReports> {
    @Nested
    private final CodeNarcReportsImpl reports

    @Inject
    protected StyledTextOutputFactory getTextOutputFactory() {
        null // see http://gradle.1045684.n5.nabble.com/injecting-dependencies-into-task-instances-td5712637.html
    }

    /**
     * Whether or not the build should break when the verifications performed by this task fail.
     */
    boolean ignoreFailures

    GradleLintReportTask() {
        reports = instantiator.newInstance(CodeNarcReportsImpl, this)
        outputs.upToDateWhen { false }
    }

    @TaskAction
    void generateReport() {
        if(reports.enabled) {
            def lintService = new LintService()
            def results = lintService.lint(project)
            def violationCount = results.violations.size()
            def textOutput = textOutputFactory.create(GradleLintReportTask)

            textOutput.text('Generated a report containing information about ')
            textOutput.style(Styling.Bold).text("$violationCount lint violation${violationCount == 1 ? '' : 's'}")
            textOutput.println(' in this project')

            reports.enabled.each { Report r ->
                ReportWriter writer = null

                switch (r.name) {
                    case 'xml': writer = new XmlReportWriter(outputFile: r.destination); break
                    case 'html': writer = new HtmlReportWriter(outputFile: r.destination); break
                    case 'text': writer = new TextReportWriter(outputFile: r.destination); break
                }

                writer.writeReport(new AnalysisContext(ruleSet: lintService.ruleSet(project)), results)
            }
        }
    }

    @Inject
    Instantiator getInstantiator() {
        null // see http://gradle.1045684.n5.nabble.com/injecting-dependencies-into-task-instances-td5712637.html
    }

    /**
     * Returns the reports to be generated by this task.
     */
    @Override
    CodeNarcReports getReports() {
        reports
    }

    /**
     * Configures the reports to be generated by this task.
     */
    @Override
    CodeNarcReports reports(Closure closure) {
        reports.configure(closure)
    }
}
