package netflix.nebula.dependency.recommender;

import netflix.nebula.dependency.recommender.provider.RecommendationProviderContainer;
import netflix.nebula.dependency.recommender.provider.RecommendationResolver;
import netflix.nebula.dependency.recommender.publisher.MavenBomXmlGenerator;
import org.codehaus.groovy.runtime.MethodClosure;
import org.gradle.api.Action;
import org.gradle.api.Plugin;
import org.gradle.api.Project;
import org.gradle.api.artifacts.*;
import org.gradle.api.plugins.ExtraPropertiesExtension;
import org.gradle.api.plugins.JavaPlugin;

import java.util.ArrayList;
import java.util.List;

public class DependencyRecommendationsPlugin implements Plugin<Project> {
    @Override
    public void apply(final Project project) {
        project.getExtensions().create("dependencyRecommendations", RecommendationProviderContainer.class, project);
        applyRecommendations(project);
        enhanceDependenciesWithRecommender(project);
        enhancePublicationsWithBomProducer(project);
    }

    private void applyRecommendations(final Project project) {
        project.getPlugins().withType(JavaPlugin.class, new Action<JavaPlugin>() {
            @Override
            public void execute(JavaPlugin javaPlugin) {
                project.getConfigurations().all(new Action<Configuration>() {
                    @Override
                    public void execute(final Configuration conf) {
                        final List<String> firstOrderDepsWithoutVersions = new ArrayList<>();

                        conf.getIncoming().beforeResolve(new Action<ResolvableDependencies>() {
                            @Override
                            public void execute(ResolvableDependencies resolvableDependencies) {
                                for (Dependency dependency : resolvableDependencies.getDependencies()) {
                                    if (dependency.getVersion() == null || dependency.getVersion().isEmpty())
                                        firstOrderDepsWithoutVersions.add(dependency.getGroup() + ":" + dependency.getName());
                                }
                            }
                        });

                        conf.getResolutionStrategy().eachDependency(new Action<DependencyResolveDetails>() {
                            @Override
                            public void execute(DependencyResolveDetails details) {
                                ModuleVersionSelector requested = details.getRequested();
                                String coord = requested.getGroup() + ":" + requested.getName();

                                // don't interfere with the way forces trump everything
                                for (ModuleVersionSelector force : conf.getResolutionStrategy().getForcedModules()) {
                                    if (requested.getGroup().equals(force.getGroup()) && requested.getName().equals(force.getName())) {
                                        return;
                                    }
                                }

                                String version = getRecommendedVersionRecursive(project, requested);
                                if (version != null && firstOrderDepsWithoutVersions.contains(coord)) {
                                    details.useVersion(version);
                                }
                            }
                        });
                    }
                });
            }
        });
    }

    protected void enhanceDependenciesWithRecommender(Project project) {
        RecommendationResolver resolver = new RecommendationResolver(project);
        project.getExtensions().getByType(ExtraPropertiesExtension.class).set("recommend",
                new MethodClosure(resolver, "recommend"));
    }

    protected void enhancePublicationsWithBomProducer(Project project) {
        project.getExtensions().create("dependencyManagement", MavenBomXmlGenerator.class, project);
    }

    /**
     * Look for recommended versions in a project and each of its ancestors in order until one is found or the root is reached
     * @return the recommended version or <code>null</code>
     */
    protected String getRecommendedVersionRecursive(Project project, ModuleVersionSelector mvSelector) {
        String version = project.getExtensions().getByType(RecommendationProviderContainer.class)
                .getRecommendedVersion(mvSelector.getGroup(), mvSelector.getName());
        if (version != null)
            return version;
        if (project.getParent() != null)
            return getRecommendedVersionRecursive(project.getParent(), mvSelector);
        return null;
    }
}
