package nebula.plugin.publishing.manifest

import nebula.plugin.info.InfoBrokerPlugin
import nebula.plugin.publishing.maven.NebulaBaseMavenPublishingPlugin
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.XmlProvider
import org.gradle.api.publish.maven.MavenPublication

/**
 * Responsible for adding metadata output from the gradle-info-plugin to the maven pom file generated by the
 * nebula-maven-publishing plugin.  The current implementation adds the metadata to the pom xml's <properties/> node,
 * with each key prefixed with a 'nebula.'.  This prefix makes it clear which key/value pairs were added by this plugin.
 *
 * Here's an example:
 * <pre>
 * <?xml version="1.0" encoding="UTF-8"?>
 * <project xmlns="http://maven.apache.org/POM/4.0.0" xsi:schemaLocation="http://maven.apache.org/POM/4.0.0 http://maven.apache.org/xsd/maven-4.0.0.xsd" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">
 *    <modelVersion>4.0.0</modelVersion>
 *    <groupId>nebula.hello</groupId>
 *    <artifactId>world</artifactId>
 *    <version>1.0</version>
 *    <dependencies>
 *      <dependency>
 *         <groupId>asm</groupId>
 *         <artifactId>asm</artifactId>
 *         <version>3.1</version>
 *         <scope>runtime</scope>
 *      </dependency>
 *    </dependencies>
 *    <properties>
 *       <nebula.Manifest-Version>1.0</nebula.Manifest-Version>
 *       <nebula.Implementation-Title>nebula.hello#world;1.0</nebula.Implementation-Title>
 *       <nebula.Implementation-Version>1.0</nebula.Implementation-Version>
 *       <nebula.Built-Status>integration</nebula.Built-Status>
 *       <nebula.Built-By>mmcgarr</nebula.Built-By>
 *       <nebula.Build-Date>2014-07-03_15:33:43</nebula.Build-Date>
 *       <nebula.Gradle-Version>1.12-20140608201532+0000</nebula.Gradle-Version>
 *       <nebula.Module-Source>/build/test/nebula.plugin.responsible.NebulaMetadataPluginLauncherSpec/published-pom-contains-a-collected-property</nebula.Module-Source>
 *       <nebula.Module-Origin>git@github.com:nebula-plugins/nebula-project-plugin.git</nebula.Module-Origin>
 *       <nebula.Change>0ab9baf</nebula.Change>
 *       <nebula.Build-Host>localhost</nebula.Build-Host>
 *       <nebula.Build-Job>LOCAL</nebula.Build-Job>
 *       <nebula.Build-Number>LOCAL</nebula.Build-Number>
 *       <nebula.Build-Id>LOCAL</nebula.Build-Id>
 *       <nebula.Created-By>1.7.0_51-b13 (Oracle Corporation)</nebula.Created-By>
 *       <nebula.Build-Java-Version>1.7.0_51</nebula.Build-Java-Version>
 *       <nebula.X-Compile-Target-JDK>1.7</nebula.X-Compile-Target-JDK>
 *       <nebula.X-Compile-Source-JDK>1.7</nebula.X-Compile-Source-JDK>
 *    </properties>
 * </project>
 * </pre>
 *
 * @author J. Michael McGarr
 */
class NebulaPublishManifestPlugin implements Plugin<Project> {

    protected Project project

    @Override
    void apply(Project project) {

        project.plugins.withType(InfoBrokerPlugin) { InfoBrokerPlugin infoPlugin ->

            project.plugins.withType(NebulaBaseMavenPublishingPlugin) { NebulaBaseMavenPublishingPlugin mavenPlugin ->
                addMetadataToMavenPom(infoPlugin, mavenPlugin)
            }

            // add Ivy metadata updating
        }
    }

    /**
     * Adds the gradle-info-plugin's metadata to the Maven pom file.
     *
     * @param infoPlugin instance used to extract the metadata
     * @param mavenPlugin instance used to grab the pom file to update
     */
    void addMetadataToMavenPom(InfoBrokerPlugin infoPlugin, NebulaBaseMavenPublishingPlugin mavenPlugin) {
        mavenPlugin.withMavenPublication { MavenPublication mavenPublication ->
            mavenPublication.pom.withXml { XmlProvider xmlProvider ->
                Map<String, String> metadata = infoPlugin.buildManifest()

                Node root = xmlProvider.asNode()
                Node propertyNode = root.children().find { it.name == 'properties' }
                if (propertyNode == null) {
                    propertyNode = root.appendNode('properties')
                }
                metadata.each { key, value ->
                    propertyNode.appendNode(scrubElementName("nebula_$key"), value)
                }
            }
        }
    }

    String scrubElementName(String name) {
        name.replaceAll(/\.|-/, '_').replaceAll(/\s/, '').trim()
    }
}
