/*
 * Copyright 2015-2017 Netflix, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package nebula.plugin.publishing.ivy

import org.gradle.api.BuildCancelledException
import org.gradle.api.Project
import org.gradle.api.XmlProvider
import org.gradle.api.artifacts.ModuleVersionIdentifier
import org.gradle.api.internal.artifacts.ivyservice.ivyresolve.strategy.DefaultVersionComparator
import org.gradle.api.internal.artifacts.ivyservice.ivyresolve.strategy.DefaultVersionSelectorScheme
import org.gradle.api.internal.artifacts.ivyservice.ivyresolve.strategy.ExactVersionSelector
import org.gradle.api.internal.artifacts.ivyservice.ivyresolve.strategy.SubVersionSelector
import org.gradle.api.internal.artifacts.ivyservice.ivyresolve.strategy.VersionSelector
import org.gradle.api.plugins.JavaBasePlugin
import org.gradle.api.publish.ivy.IvyPublication

/**
 * Replaces first order dependencies with the selected versions when publishing.
 */
class IvyResolvedDependenciesPlugin extends AbstractResolvedDependenciesPlugin {
    @Override
    void apply(Project project) {
        project.plugins.apply IvyBasePublishPlugin

        project.publishing {
            publications {
                withType(IvyPublication) {
                    descriptor.withXml { XmlProvider xml ->
                        project.plugins.withType(JavaBasePlugin) {
                            def dependencies = xml.asNode()?.dependencies?.dependency
                            dependencies?.each { Node dep ->
                                String scope = dep.@conf
                                String group = dep.@org
                                String name = dep.@name

                                if (scope == 'compile->default') {
                                    scope = 'compile'
                                }

                                if (scope == 'provided->default' || scope == 'runtime->default') {
                                    scope = 'runtime'
                                }

                                if (scope == 'test->default') {
                                    scope = 'test'
                                }

                                if (scope.contains('->')) {
                                    scope = scope.split('->')[0]
                                }

                                def mvid = selectedModuleVersion(project, scope, group, name)
                                if (!mvid) {
                                    return  // continue loop if a dependency is not found in dependencyMap
                                }

                                if (dep.@rev) {
                                    def version = dep.@rev as String
                                    VersionSelector selector = parseSelector(version)
                                    verifySubVersion(selector, mvid)
                                    setVersionConstraint(selector, version, dep)
                                }

                                dep.@org = mvid.group
                                dep.@name = mvid.name
                                dep.@rev = mvid.version
                            }
                        }
                    }
                }
            }
        }
    }

    private VersionSelector parseSelector(String version) {
        def scheme = new DefaultVersionSelectorScheme(new DefaultVersionComparator())
        def selector = scheme.parseSelector(version)
        selector
    }

    private void verifySubVersion(VersionSelector selector, ModuleVersionIdentifier mvid) {
        if (selector instanceof SubVersionSelector && !selector.selector.endsWith(".+")) {
            throw new BuildCancelledException(
                    "Incorrect version definition detected.\n" +
                            "Dependency '${mvid.group}:${mvid.name}:${selector.selector}' has version definition which" +
                            " resolves into unexpected version.\n" +
                            "\n" +
                            "E.g. 1.1+ resolves to 1.1, 1.10, 1.11 etc but misses 1.2, 1.3.\n" +
                            "\n" +
                            "We recommend to use definition like 1.+ for the highest from major version 1 or " +
                            "[1.1,] which is anything equal or higher then 1.1.\n")
        }
    }

    private void setVersionConstraint(VersionSelector selector, String version, Node dep) {
        if (!(selector instanceof ExactVersionSelector)) {
            dep.@revConstraint = version
        }
    }
}
