/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.loadbalancer;

import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.netflix.client.IClientConfigAware;
import com.netflix.client.config.CommonClientConfigKey;
import com.netflix.client.config.IClientConfig;
import com.netflix.client.config.IClientConfigKey;
import com.netflix.client.config.UnboxedIntProperty;
import com.netflix.loadbalancer.Server;
import com.netflix.loadbalancer.ServerStats;
import com.netflix.loadbalancer.ZoneSnapshot;
import com.netflix.loadbalancer.ZoneStats;
import com.netflix.servo.annotations.DataSourceType;
import com.netflix.servo.annotations.Monitor;
import com.netflix.servo.monitor.Monitors;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

public class LoadBalancerStats
implements IClientConfigAware {
    private static final String PREFIX = "LBStats_";
    public static final IClientConfigKey<Integer> ACTIVE_REQUESTS_COUNT_TIMEOUT = new CommonClientConfigKey<Integer>("niws.loadbalancer.serverStats.activeRequestsCount.effectiveWindowSeconds", Integer.valueOf(600)){};
    public static final IClientConfigKey<Integer> CONNECTION_FAILURE_COUNT_THRESHOLD = new CommonClientConfigKey<Integer>("niws.loadbalancer.%s.connectionFailureCountThreshold", Integer.valueOf(3)){};
    public static final IClientConfigKey<Integer> CIRCUIT_TRIP_TIMEOUT_FACTOR_SECONDS = new CommonClientConfigKey<Integer>("niws.loadbalancer.%s.circuitTripTimeoutFactorSeconds", Integer.valueOf(10)){};
    public static final IClientConfigKey<Integer> CIRCUIT_TRIP_MAX_TIMEOUT_SECONDS = new CommonClientConfigKey<Integer>("niws.loadbalancer.%s.circuitTripMaxTimeoutSeconds", Integer.valueOf(30)){};
    private String name;
    volatile Map<String, ZoneStats> zoneStatsMap = new ConcurrentHashMap<String, ZoneStats>();
    volatile Map<String, List<? extends Server>> upServerListZoneMap = new ConcurrentHashMap<String, List<? extends Server>>();
    private UnboxedIntProperty connectionFailureThreshold = new UnboxedIntProperty(((Integer)CONNECTION_FAILURE_COUNT_THRESHOLD.defaultValue()).intValue());
    private UnboxedIntProperty circuitTrippedTimeoutFactor = new UnboxedIntProperty(((Integer)CIRCUIT_TRIP_TIMEOUT_FACTOR_SECONDS.defaultValue()).intValue());
    private UnboxedIntProperty maxCircuitTrippedTimeout = new UnboxedIntProperty(((Integer)CIRCUIT_TRIP_MAX_TIMEOUT_SECONDS.defaultValue()).intValue());
    private UnboxedIntProperty activeRequestsCountTimeout = new UnboxedIntProperty(((Integer)ACTIVE_REQUESTS_COUNT_TIMEOUT.defaultValue()).intValue());
    private final LoadingCache<Server, ServerStats> serverStatsCache = CacheBuilder.newBuilder().expireAfterAccess(30L, TimeUnit.MINUTES).removalListener(notification -> ((ServerStats)notification.getValue()).close()).build((CacheLoader)new CacheLoader<Server, ServerStats>(){

        public ServerStats load(Server server) {
            return LoadBalancerStats.this.createServerStats(server);
        }
    });
    private static Comparator<ServerStats> serverStatsComparator = new Comparator<ServerStats>(){

        @Override
        public int compare(ServerStats o1, ServerStats o2) {
            String zone1 = "";
            String zone2 = "";
            if (o1.server != null && o1.server.getZone() != null) {
                zone1 = o1.server.getZone();
            }
            if (o2.server != null && o2.server.getZone() != null) {
                zone2 = o2.server.getZone();
            }
            return zone1.compareTo(zone2);
        }
    };

    protected ServerStats createServerStats(Server server) {
        ServerStats ss = new ServerStats(this);
        ss.setBufferSize(1000);
        ss.setPublishInterval(1000);
        ss.initialize(server);
        return ss;
    }

    public LoadBalancerStats() {
    }

    public LoadBalancerStats(String name) {
        this.name = name;
        Monitors.registerObject((String)name, (Object)this);
    }

    public void initWithNiwsConfig(IClientConfig clientConfig) {
        this.name = clientConfig.getClientName();
        Preconditions.checkArgument((this.name != null ? 1 : 0) != 0, (Object)"IClientConfig#getCLientName() must not be null");
        this.connectionFailureThreshold = new UnboxedIntProperty(clientConfig.getGlobalProperty(CONNECTION_FAILURE_COUNT_THRESHOLD.format(new Object[]{this.name})));
        this.circuitTrippedTimeoutFactor = new UnboxedIntProperty(clientConfig.getGlobalProperty(CIRCUIT_TRIP_TIMEOUT_FACTOR_SECONDS.format(new Object[]{this.name})));
        this.maxCircuitTrippedTimeout = new UnboxedIntProperty(clientConfig.getGlobalProperty(CIRCUIT_TRIP_MAX_TIMEOUT_SECONDS.format(new Object[]{this.name})));
        this.activeRequestsCountTimeout = new UnboxedIntProperty(clientConfig.getGlobalProperty(ACTIVE_REQUESTS_COUNT_TIMEOUT));
    }

    public String getName() {
        return this.name;
    }

    public void setName(String name) {
        this.name = name;
    }

    UnboxedIntProperty getConnectionFailureCountThreshold() {
        return this.connectionFailureThreshold;
    }

    UnboxedIntProperty getCircuitTrippedTimeoutFactor() {
        return this.circuitTrippedTimeoutFactor;
    }

    UnboxedIntProperty getCircuitTripMaxTimeoutSeconds() {
        return this.maxCircuitTrippedTimeout;
    }

    UnboxedIntProperty getActiveRequestsCountTimeout() {
        return this.activeRequestsCountTimeout;
    }

    public void updateServerList(List<Server> servers) {
        for (Server s : servers) {
            this.addServer(s);
        }
    }

    public void addServer(Server server) {
        if (server != null) {
            try {
                this.serverStatsCache.get((Object)server);
            }
            catch (ExecutionException e) {
                ServerStats stats = this.createServerStats(server);
                this.serverStatsCache.asMap().putIfAbsent(server, stats);
            }
        }
    }

    public void noteResponseTime(Server server, double msecs) {
        ServerStats ss = this.getServerStats(server);
        ss.noteResponseTime(msecs);
    }

    protected ServerStats getServerStats(Server server) {
        if (server == null) {
            return null;
        }
        try {
            return (ServerStats)this.serverStatsCache.get((Object)server);
        }
        catch (ExecutionException e) {
            ServerStats stats = this.createServerStats(server);
            this.serverStatsCache.asMap().putIfAbsent(server, stats);
            return (ServerStats)this.serverStatsCache.asMap().get(server);
        }
    }

    public void incrementActiveRequestsCount(Server server) {
        ServerStats ss = this.getServerStats(server);
        ss.incrementActiveRequestsCount();
    }

    public void decrementActiveRequestsCount(Server server) {
        ServerStats ss = this.getServerStats(server);
        ss.decrementActiveRequestsCount();
    }

    private ZoneStats getZoneStats(String zone) {
        ZoneStats zs = this.zoneStatsMap.get(zone = zone.toLowerCase());
        if (zs == null) {
            this.zoneStatsMap.put(zone, new ZoneStats(this.getName(), zone, this));
            zs = this.zoneStatsMap.get(zone);
        }
        return zs;
    }

    public boolean isCircuitBreakerTripped(Server server) {
        ServerStats ss = this.getServerStats(server);
        return ss.isCircuitBreakerTripped();
    }

    public void incrementSuccessiveConnectionFailureCount(Server server) {
        ServerStats ss = this.getServerStats(server);
        ss.incrementSuccessiveConnectionFailureCount();
    }

    public void clearSuccessiveConnectionFailureCount(Server server) {
        ServerStats ss = this.getServerStats(server);
        ss.clearSuccessiveConnectionFailureCount();
    }

    public void incrementNumRequests(Server server) {
        ServerStats ss = this.getServerStats(server);
        ss.incrementNumRequests();
    }

    public void incrementZoneCounter(Server server) {
        String zone = server.getZone();
        if (zone != null) {
            this.getZoneStats(zone).incrementCounter();
        }
    }

    public void updateZoneServerMapping(Map<String, List<Server>> map) {
        this.upServerListZoneMap = new ConcurrentHashMap<String, List<Server>>(map);
        for (String zone : map.keySet()) {
            this.getZoneStats(zone);
        }
    }

    public int getInstanceCount(String zone) {
        if (zone == null) {
            return 0;
        }
        List<? extends Server> currentList = this.upServerListZoneMap.get(zone = zone.toLowerCase());
        if (currentList == null) {
            return 0;
        }
        return currentList.size();
    }

    public int getActiveRequestsCount(String zone) {
        return this.getZoneSnapshot(zone).getActiveRequestsCount();
    }

    public double getActiveRequestsPerServer(String zone) {
        return this.getZoneSnapshot(zone).getLoadPerServer();
    }

    public ZoneSnapshot getZoneSnapshot(String zone) {
        if (zone == null) {
            return new ZoneSnapshot();
        }
        zone = zone.toLowerCase();
        List<? extends Server> currentList = this.upServerListZoneMap.get(zone);
        return this.getZoneSnapshot(currentList);
    }

    public ZoneSnapshot getZoneSnapshot(List<? extends Server> servers) {
        if (servers == null || servers.size() == 0) {
            return new ZoneSnapshot();
        }
        int instanceCount = servers.size();
        int activeConnectionsCount = 0;
        int activeConnectionsCountOnAvailableServer = 0;
        int circuitBreakerTrippedCount = 0;
        double loadPerServer = 0.0;
        long currentTime = System.currentTimeMillis();
        for (Server server : servers) {
            ServerStats stat = this.getSingleServerStat(server);
            if (stat.isCircuitBreakerTripped(currentTime)) {
                ++circuitBreakerTrippedCount;
            } else {
                activeConnectionsCountOnAvailableServer += stat.getActiveRequestsCount(currentTime);
            }
            activeConnectionsCount += stat.getActiveRequestsCount(currentTime);
        }
        if (circuitBreakerTrippedCount == instanceCount) {
            if (instanceCount > 0) {
                loadPerServer = -1.0;
            }
        } else {
            loadPerServer = (double)activeConnectionsCountOnAvailableServer / (double)(instanceCount - circuitBreakerTrippedCount);
        }
        return new ZoneSnapshot(instanceCount, circuitBreakerTrippedCount, activeConnectionsCount, loadPerServer);
    }

    public int getCircuitBreakerTrippedCount(String zone) {
        return this.getZoneSnapshot(zone).getCircuitTrippedCount();
    }

    @Monitor(name="LBStats_CircuitBreakerTrippedCount", type=DataSourceType.GAUGE)
    public int getCircuitBreakerTrippedCount() {
        int count = 0;
        for (String zone : this.upServerListZoneMap.keySet()) {
            count += this.getCircuitBreakerTrippedCount(zone);
        }
        return count;
    }

    public long getMeasuredZoneHits(String zone) {
        if (zone == null) {
            return 0L;
        }
        zone = zone.toLowerCase();
        long count = 0L;
        List<? extends Server> currentList = this.upServerListZoneMap.get(zone);
        if (currentList == null) {
            return 0L;
        }
        for (Server server : currentList) {
            ServerStats stat = this.getSingleServerStat(server);
            count += stat.getMeasuredRequestsCount();
        }
        return count;
    }

    public int getCongestionRatePercentage(String zone) {
        if (zone == null) {
            return 0;
        }
        List<? extends Server> currentList = this.upServerListZoneMap.get(zone = zone.toLowerCase());
        if (currentList == null || currentList.size() == 0) {
            return 0;
        }
        int serverCount = currentList.size();
        int activeConnectionsCount = 0;
        int circuitBreakerTrippedCount = 0;
        for (Server server : currentList) {
            ServerStats stat = this.getSingleServerStat(server);
            activeConnectionsCount += stat.getActiveRequestsCount();
            if (!stat.isCircuitBreakerTripped()) continue;
            ++circuitBreakerTrippedCount;
        }
        return (int)((long)(activeConnectionsCount + circuitBreakerTrippedCount) * 100L / (long)serverCount);
    }

    @Monitor(name="LBStats_AvailableZones", type=DataSourceType.INFORMATIONAL)
    public Set<String> getAvailableZones() {
        return this.upServerListZoneMap.keySet();
    }

    public ServerStats getSingleServerStat(Server server) {
        return this.getServerStats(server);
    }

    public Map<Server, ServerStats> getServerStats() {
        return this.serverStatsCache.asMap();
    }

    public Map<String, ZoneStats> getZoneStats() {
        return this.zoneStatsMap;
    }

    public String toString() {
        return "Zone stats: " + this.zoneStatsMap.toString() + ",Server stats: " + LoadBalancerStats.getSortedServerStats(this.getServerStats().values()).toString();
    }

    private static Collection<ServerStats> getSortedServerStats(Collection<ServerStats> stats) {
        ArrayList<ServerStats> list = new ArrayList<ServerStats>(stats);
        Collections.sort(list, serverStatsComparator);
        return list;
    }
}

