/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.spectator.impl;

import com.netflix.spectator.api.Clock;
import com.netflix.spectator.api.Counter;
import com.netflix.spectator.api.Id;
import com.netflix.spectator.api.Registry;
import com.netflix.spectator.api.Timer;
import com.netflix.spectator.api.patterns.PolledMeter;
import java.time.Duration;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Scheduler {
    private static final Logger LOGGER = LoggerFactory.getLogger(Scheduler.class);
    private final DelayQueue<DelayedTask> queue = new DelayQueue();
    private final Clock clock;
    private final Stats stats;
    private final ThreadFactory factory;
    private final Thread[] threads;
    private final Lock lock = new ReentrantLock();
    private volatile boolean started = false;
    private volatile boolean shutdown = false;

    private static ThreadFactory newThreadFactory(final String id) {
        return new ThreadFactory(){
            private final AtomicInteger next = new AtomicInteger();

            @Override
            public Thread newThread(Runnable r) {
                String name = "spectator-" + id + "-" + this.next.getAndIncrement();
                Thread t = new Thread(r, name);
                t.setDaemon(true);
                return t;
            }
        };
    }

    private static Id newId(Registry registry, String id, String name) {
        return registry.createId("spectator.scheduler." + name, "id", id);
    }

    public Scheduler(Registry registry, String id, int poolSize) {
        this.clock = registry.clock();
        PolledMeter.using(registry).withId(Scheduler.newId(registry, id, "queueSize")).monitorSize(this.queue);
        this.stats = new Stats(registry, id);
        this.factory = Scheduler.newThreadFactory(id);
        this.threads = new Thread[poolSize];
    }

    public ScheduledFuture<?> schedule(Options options, Runnable task) {
        if (!this.started) {
            this.startThreads();
        }
        DelayedTask t = new DelayedTask(this.clock, options, task);
        this.queue.put(t);
        return t;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void shutdown() {
        this.lock.lock();
        try {
            this.shutdown = true;
            for (Thread thread : this.threads) {
                if (thread == null || !thread.isAlive()) continue;
                thread.interrupt();
            }
            for (int i = 0; i < this.threads.length; ++i) {
                if (this.threads[i] == null) continue;
                try {
                    this.threads[i].join();
                }
                catch (Exception e) {
                    LOGGER.debug("exception while shutting down thread {}", (Object)this.threads[i].getName(), (Object)e);
                }
                this.threads[i] = null;
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    private void startThreads() {
        if (this.shutdown) {
            return;
        }
        this.lock.lock();
        try {
            if (!this.shutdown) {
                this.started = true;
                for (int i = 0; i < this.threads.length; ++i) {
                    if (this.threads[i] != null && this.threads[i].isAlive() && !this.threads[i].isInterrupted()) continue;
                    this.threads[i] = this.factory.newThread(new Worker());
                    this.threads[i].start();
                    LOGGER.debug("started thread {}", (Object)this.threads[i].getName());
                }
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    private final class Worker
    implements Runnable {
        private Worker() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            try {
                while (!Scheduler.this.shutdown && !Thread.currentThread().isInterrupted()) {
                    try {
                        DelayedTask task = (DelayedTask)Scheduler.this.queue.take();
                        Scheduler.this.stats.incrementActiveTaskCount();
                        long delay = Scheduler.this.clock.wallTime() - task.getNextExecutionTime();
                        Scheduler.this.stats.taskExecutionDelay().record(delay, TimeUnit.MILLISECONDS);
                        Scheduler.this.stats.taskExecutionTime().recordRunnable(() -> task.runAndReschedule(Scheduler.this.queue, Scheduler.this.stats));
                    }
                    catch (InterruptedException e) {
                        LOGGER.debug("task interrupted", (Throwable)e);
                        break;
                    }
                    finally {
                        Scheduler.this.stats.decrementActiveTaskCount();
                    }
                }
            }
            finally {
                Scheduler.this.startThreads();
            }
        }
    }

    static class DelayedTask
    implements ScheduledFuture<Void> {
        private final Clock clock;
        private final Options options;
        private final Runnable task;
        private final long initialExecutionTime;
        private volatile long nextExecutionTime;
        private volatile Thread thread = null;
        private volatile boolean cancelled = false;

        DelayedTask(Clock clock, Options options, Runnable task) {
            this.clock = clock;
            this.options = options;
            this.task = task;
            this.nextExecutionTime = this.initialExecutionTime = clock.wallTime() + options.initialDelay;
        }

        long getNextExecutionTime() {
            return this.nextExecutionTime;
        }

        void updateNextExecutionTime(Counter skipped) {
            long nextTime = this.nextExecutionTime;
            switch (this.options.schedulingPolicy) {
                case FIXED_DELAY: {
                    nextTime = this.clock.wallTime() + this.options.frequencyMillis;
                    break;
                }
                case FIXED_RATE_SKIP_IF_LONG: {
                    long now = this.clock.wallTime();
                    nextTime += this.options.frequencyMillis;
                    while (nextTime < now) {
                        nextTime += this.options.frequencyMillis;
                        skipped.increment();
                    }
                    break;
                }
            }
            this.nextExecutionTime = nextTime;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void runAndReschedule(DelayQueue<DelayedTask> queue, Stats stats) {
            this.thread = Thread.currentThread();
            boolean scheduleAgain = this.options.schedulingPolicy != Policy.RUN_ONCE;
            try {
                if (!this.isDone()) {
                    this.task.run();
                }
            }
            catch (Throwable t) {
                LOGGER.warn("task execution failed", t);
                stats.incrementUncaught(t);
                scheduleAgain = !this.options.stopOnFailure;
            }
            finally {
                this.thread = null;
                if (scheduleAgain && !this.isDone()) {
                    this.updateNextExecutionTime(stats.skipped());
                    queue.put(this);
                } else {
                    this.cancelled = true;
                }
            }
        }

        @Override
        public long getDelay(TimeUnit unit) {
            long delayMillis = Math.max(this.nextExecutionTime - this.clock.wallTime(), 0L);
            return unit.convert(delayMillis, TimeUnit.MILLISECONDS);
        }

        @Override
        public int compareTo(Delayed other) {
            long d1 = this.getDelay(TimeUnit.MILLISECONDS);
            long d2 = other.getDelay(TimeUnit.MILLISECONDS);
            return Long.compare(d1, d2);
        }

        @Override
        public boolean cancel(boolean mayInterruptIfRunning) {
            this.cancelled = true;
            Thread t = this.thread;
            if (mayInterruptIfRunning && t != null) {
                t.interrupt();
            }
            return true;
        }

        @Override
        public boolean isCancelled() {
            return this.cancelled;
        }

        @Override
        public boolean isDone() {
            return this.cancelled;
        }

        @Override
        public Void get() throws InterruptedException, ExecutionException {
            throw new UnsupportedOperationException();
        }

        @Override
        public Void get(long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException {
            throw new UnsupportedOperationException();
        }
    }

    static class Stats {
        private final Registry registry;
        private final AtomicInteger activeCount;
        private final Timer taskExecutionTime;
        private final Timer taskExecutionDelay;
        private final Counter skipped;
        private final Id uncaughtExceptionsId;

        Stats(Registry registry, String id) {
            this.registry = registry;
            this.activeCount = PolledMeter.using(registry).withId(Scheduler.newId(registry, id, "activeThreads")).monitorValue(new AtomicInteger());
            this.taskExecutionTime = registry.timer(Scheduler.newId(registry, id, "taskExecutionTime"));
            this.taskExecutionDelay = registry.timer(Scheduler.newId(registry, id, "taskExecutionDelay"));
            this.skipped = registry.counter(Scheduler.newId(registry, id, "skipped"));
            this.uncaughtExceptionsId = Scheduler.newId(registry, id, "uncaughtExceptions");
        }

        void incrementActiveTaskCount() {
            this.activeCount.incrementAndGet();
        }

        void decrementActiveTaskCount() {
            this.activeCount.decrementAndGet();
        }

        Timer taskExecutionTime() {
            return this.taskExecutionTime;
        }

        Timer taskExecutionDelay() {
            return this.taskExecutionDelay;
        }

        Counter skipped() {
            return this.skipped;
        }

        void incrementUncaught(Throwable t) {
            String cls = t.getClass().getSimpleName();
            this.registry.counter(this.uncaughtExceptionsId.withTag("exception", cls)).increment();
        }
    }

    public static class Options {
        private Policy schedulingPolicy = Policy.RUN_ONCE;
        private long initialDelay = 0L;
        private long frequencyMillis = 0L;
        private boolean stopOnFailure = false;

        public Options withInitialDelay(Duration delay) {
            this.initialDelay = delay.toMillis();
            return this;
        }

        public Options withFrequency(Policy policy, Duration frequency) {
            this.schedulingPolicy = policy;
            this.frequencyMillis = frequency.toMillis();
            return this;
        }

        public Options withStopOnFailure(boolean flag) {
            this.stopOnFailure = flag;
            return this;
        }
    }

    public static enum Policy {
        RUN_ONCE,
        FIXED_DELAY,
        FIXED_RATE_SKIP_IF_LONG;

    }
}

