/*
 * Decompiled with CFR 0.152.
 */
package com.netgrif.application.engine.elastic.service;

import com.netgrif.application.engine.elastic.domain.ElasticJob;
import com.netgrif.application.engine.elastic.domain.ElasticTask;
import com.netgrif.application.engine.elastic.domain.ElasticTaskJob;
import com.netgrif.application.engine.elastic.domain.ElasticTaskRepository;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import javax.annotation.PreDestroy;
import lombok.Generated;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.dao.InvalidDataAccessApiUsageException;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Service;

@Service
public class ElasticTaskQueueManager {
    @Generated
    private static final Logger log = LoggerFactory.getLogger(ElasticTaskQueueManager.class);
    private final ThreadPoolTaskExecutor elasticTaskExecutor;
    private final ElasticTaskRepository repository;
    private final ConcurrentHashMap<String, BlockingQueue<Runnable>> taskQueues = new ConcurrentHashMap();
    private final Set<String> activeTasks = ConcurrentHashMap.newKeySet();

    public ElasticTaskQueueManager(@Qualifier(value="elasticTaskExecutor") ThreadPoolTaskExecutor elasticTaskExecutor, ElasticTaskRepository repository) {
        this.elasticTaskExecutor = elasticTaskExecutor;
        this.repository = repository;
    }

    public Future<ElasticTask> scheduleOperation(ElasticTaskJob task) {
        if (task.getTask().getTaskId() == null) {
            throw new IllegalArgumentException("Task id cannot be null");
        }
        String taskId = task.getTaskId();
        log.debug("Scheduling operation for task: {}", (Object)taskId);
        CompletableFuture<ElasticTask> future = new CompletableFuture<ElasticTask>();
        Runnable taskWrapper = () -> {
            try {
                ElasticTask result = this.processTask(task);
                future.complete(result);
            }
            catch (Exception e) {
                future.completeExceptionally(e);
            }
        };
        BlockingQueue queue = this.taskQueues.computeIfAbsent(taskId, k -> new LinkedBlockingQueue());
        try {
            queue.add(taskWrapper);
        }
        catch (Exception e) {
            log.error("Queue error:" + e.getMessage());
            throw e;
        }
        if (this.activeTasks.add(taskId)) {
            log.debug("Task {} is ready for processing. Submitting to executor.", (Object)taskId);
            this.elasticTaskExecutor.submit(Objects.requireNonNull((Runnable)queue.poll()));
        } else {
            log.debug("Task {} is queued for processing.", (Object)taskId);
        }
        return future;
    }

    private ElasticTask processTask(ElasticTaskJob task) {
        try {
            log.debug("Processing task: {}", (Object)task.getTaskId());
            switch (task.getJobType()) {
                case INDEX: {
                    ElasticTask elasticTask = this.indexTaskWorker(task.getTask());
                    return elasticTask;
                }
                case REMOVE: {
                    ElasticTask elasticTask = this.removeTaskWorker(task.getTask());
                    return elasticTask;
                }
            }
            try {
                log.warn("Unknown job type for task: {}", (Object)task.getTaskId());
                throw new IllegalArgumentException("Unknown job type: " + task.getJobType());
            }
            catch (Exception e) {
                log.error("Error processing task {}: {}", new Object[]{task.getTaskId(), e.getMessage(), e});
                throw e;
            }
        }
        finally {
            this.activeTasks.remove(task.getTaskId());
            this.scheduleNextTask(task.getTaskId());
        }
    }

    private void scheduleNextTask(String taskId) {
        BlockingQueue<Runnable> queue = this.taskQueues.get(taskId);
        if (queue != null) {
            Runnable nextTask = (Runnable)queue.poll();
            if (nextTask != null) {
                log.debug("Submitting next task for ID: {} to executor", (Object)taskId);
                this.elasticTaskExecutor.submit(nextTask);
            } else {
                this.activeTasks.remove(taskId);
                this.taskQueues.remove(taskId);
            }
        }
    }

    @PreDestroy
    public void destroy() throws InterruptedException {
        log.info("Shutting down ElasticTaskQueueManager");
        this.elasticTaskExecutor.shutdown();
    }

    private ElasticTask indexTaskWorker(ElasticTask task) {
        log.debug("Indexing task [{}] in thread [{}]", (Object)task.getTaskId(), (Object)Thread.currentThread().getName());
        ElasticTask elasticTask = null;
        try {
            elasticTask = this.repository.findByStringId(task.getStringId());
            if (elasticTask == null) {
                elasticTask = (ElasticTask)this.repository.save(task);
            } else {
                elasticTask.update(task);
                elasticTask = (ElasticTask)this.repository.save(elasticTask);
            }
            log.debug("[{}]: Task \"{}\" [{}] indexed", new Object[]{task.getCaseId(), task.getTitle(), task.getStringId()});
        }
        catch (InvalidDataAccessApiUsageException e) {
            log.debug("[{}]: Task \"{}\" has duplicates, will be reindexed", (Object)task.getCaseId(), (Object)task.getTitle());
            this.repository.deleteAllByStringId(task.getStringId());
            this.repository.save(task);
            log.debug("[{}]: Task \"{}\" indexed", (Object)task.getCaseId(), (Object)task.getTitle());
        }
        catch (RuntimeException e) {
            log.error("Elastic executor was killed before finish: {}", (Object)e.getMessage());
        }
        return elasticTask;
    }

    private ElasticTask removeTaskWorker(ElasticTask task) {
        log.debug("Remove task [{}] in thread [{}]", (Object)task.getTaskId(), (Object)Thread.currentThread().getName());
        try {
            log.debug("[{}]: Task \"{}\" [{}] removed", new Object[]{task.getCaseId(), task.getTitle(), task.getStringId()});
            return this.repository.deleteAllByTaskId(task.getTaskId());
        }
        catch (RuntimeException e) {
            log.error("Elastic executor was killed before finish: {}", (Object)e.getMessage());
            return task;
        }
    }

    public void removeTasksByProcess(String processId) {
        List<ElasticTask> tasks = this.repository.findAllByProcessId(processId);
        long maxWaitTime = 30L;
        long baseWaitTime = 1L;
        tasks.forEach(task -> {
            ElasticTaskJob job = new ElasticTaskJob(ElasticJob.REMOVE, (ElasticTask)task);
            CompletableFuture<ElasticTask> removeJobFuture = CompletableFuture.supplyAsync(() -> this.processTask(job), (Executor)this.elasticTaskExecutor);
            long waitTime = baseWaitTime;
            while (true) {
                try {
                    removeJobFuture.get(waitTime, TimeUnit.SECONDS);
                }
                catch (TimeoutException e) {
                    if (waitTime >= maxWaitTime) {
                        log.error("Timeout: Task {} did not complete within {} seconds", (Object)task.getTaskId(), (Object)maxWaitTime);
                        break;
                    }
                    waitTime *= 2L;
                    continue;
                }
                catch (InterruptedException | ExecutionException e) {
                    log.error("Exception during task execution: {}", (Object)e.getMessage(), (Object)e);
                }
                break;
            }
        });
    }
}

