/*
 * Decompiled with CFR 0.152.
 */
package com.newrelic.agent.android.aei;

import android.annotation.SuppressLint;
import android.app.ActivityManager;
import android.app.ApplicationExitInfo;
import android.content.Context;
import android.os.Build;
import android.os.Process;
import androidx.annotation.NonNull;
import androidx.annotation.RequiresApi;
import com.newrelic.agent.android.AgentConfiguration;
import com.newrelic.agent.android.aei.AEISessionMapper;
import com.newrelic.agent.android.aei.AEITrace;
import com.newrelic.agent.android.aei.AEITraceReporter;
import com.newrelic.agent.android.aei.Error;
import com.newrelic.agent.android.analytics.AnalyticsControllerImpl;
import com.newrelic.agent.android.harvest.Harvest;
import com.newrelic.agent.android.logging.AgentLog;
import com.newrelic.agent.android.logging.AgentLogManager;
import com.newrelic.agent.android.stats.StatsEngine;
import com.newrelic.agent.android.util.Streams;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

public class ApplicationExitMonitor {
    private static final AgentLog log = AgentLogManager.getAgentLog();
    static final String SESSION_ID_MAPPING_STORE = "sessionMeta.map";
    static final String ARTIFACT_NAME = "aei-%s.dat";
    protected final File reportsDir;
    protected final String packageName;
    protected final AEISessionMapper sessionMapper;
    protected final ActivityManager am;
    protected final AEITraceReporter traceReporter;
    private static final Map<Integer, String> REASON_MAP = new HashMap<Integer, String>();
    private static final Map<Integer, String> IMPORTANCE_MAP = new HashMap<Integer, String>();

    public ApplicationExitMonitor(Context context) {
        this.reportsDir = new File(context.getCacheDir(), "newrelic/applicationExitInfo");
        this.packageName = context.getPackageName();
        this.sessionMapper = new AEISessionMapper(new File(this.reportsDir, SESSION_ID_MAPPING_STORE));
        this.am = (ActivityManager)context.getSystemService("activity");
        this.reportsDir.mkdirs();
        AEITraceReporter traceReporter = AEITraceReporter.getInstance();
        try {
            traceReporter = AEITraceReporter.initialize(this.reportsDir, AgentConfiguration.getInstance());
            if (traceReporter != null) {
                traceReporter.start();
                if (!traceReporter.isStarted()) {
                    log.warn("ApplicationExitMonitor: AEI trace reporter not started. AEITrace reporting will be disabled.");
                }
            } else {
                log.warn("ApplicationExitMonitor: No AEI trace reporter. AEITrace reporting will be disabled.");
            }
        }
        catch (IOException e) {
            log.error("ApplicationExitMonitor: " + e);
        }
        this.traceReporter = traceReporter;
    }

    int getCurrentProcessId() {
        return Process.myPid();
    }

    @SuppressLint(value={"SwitchIntDef"})
    public void harvestApplicationExitInfo() {
        this.sessionMapper.load();
        if (Build.VERSION.SDK_INT >= 30) {
            AtomicInteger recordsVisited = new AtomicInteger(0);
            AtomicInteger recordsSkipped = new AtomicInteger(0);
            AtomicInteger recordsDropped = new AtomicInteger(0);
            if (null == this.am) {
                log.error("harvestApplicationExitInfo: ActivityManager is null! Cannot record ApplicationExitInfo data.");
                return;
            }
            List applicationExitInfoList = this.am.getHistoricalProcessExitReasons(this.packageName, 0, 0);
            for (ApplicationExitInfo exitInfo : applicationExitInfoList) {
                HashMap<String, Object> eventAttributes;
                File artifact = new File(this.reportsDir, String.format(Locale.getDefault(), ARTIFACT_NAME, exitInfo.getPid()));
                if (artifact.exists() && artifact.length() > 0L) {
                    log.debug("ApplicationExitMonitor: skipping exit info for pid[" + exitInfo.getPid() + "]: already recorded.");
                    recordsSkipped.incrementAndGet();
                    continue;
                }
                String aeiSessionId = this.sessionMapper.getSessionId(exitInfo.getPid());
                if (aeiSessionId != null && !aeiSessionId.isEmpty() && !aeiSessionId.equals(AgentConfiguration.getInstance().getSessionID())) {
                    log.debug("ApplicationExitMonitor: Found session id [" + aeiSessionId + "] for AEI pid[" + exitInfo.getPid() + "]");
                }
                String traceReport = exitInfo.toString();
                if (artifact.exists() && artifact.length() == 0L) {
                    artifact.delete();
                }
                try (FileOutputStream artifactOs = new FileOutputStream(artifact, false);){
                    if (null != exitInfo.getTraceInputStream()) {
                        try (InputStream traceIs = exitInfo.getTraceInputStream();){
                            traceReport = Streams.slurpString(traceIs);
                        }
                        catch (IOException e) {
                            log.info("ApplicationExitMonitor: " + e);
                        }
                    }
                    ((OutputStream)artifactOs).write(traceReport.getBytes(StandardCharsets.UTF_8));
                    artifactOs.flush();
                    ((OutputStream)artifactOs).close();
                    artifact.setReadOnly();
                    recordsVisited.incrementAndGet();
                }
                catch (IOException e) {
                    log.debug("harvestApplicationExitInfo: AppExitInfo artifact error. " + e);
                }
                AEISessionMapper.AEISessionMeta sessionMeta = this.sessionMapper.get(exitInfo.getPid());
                if (sessionMeta != null) {
                    log.debug("ApplicationExitMonitor: Using session meta [" + sessionMeta.sessionId + ", " + sessionMeta.realAgentId + "] for AEI pid[" + exitInfo.getPid() + "]");
                }
                try {
                    eventAttributes = this.getEventAttributesForAEI(exitInfo, sessionMeta, traceReport);
                }
                catch (UnsupportedEncodingException e) {
                    throw new RuntimeException(e);
                }
                StatsEngine.SUPPORTABILITY.inc("Supportability/AgentHealth/ApplicationExitInfo/status/" + exitInfo.getStatus());
                StatsEngine.SUPPORTABILITY.inc("Supportability/AgentHealth/ApplicationExitInfo/reason/" + this.getReasonAsString(exitInfo.getReason()));
                StatsEngine.SUPPORTABILITY.inc("Supportability/AgentHealth/ApplicationExitInfo/importance/" + this.getImportanceAsString(exitInfo.getImportance()));
                StatsEngine.SUPPORTABILITY.sample("Supportability/AgentHealth/ApplicationExitInfo/visited", recordsVisited.get());
                StatsEngine.SUPPORTABILITY.sample("Supportability/AgentHealth/ApplicationExitInfo/skipped", recordsSkipped.get());
                AnalyticsControllerImpl analyticsController = AnalyticsControllerImpl.getInstance();
                Error error = new Error(analyticsController.getSessionAttributes(), eventAttributes, sessionMeta);
                this.traceReporter.reportAEITrace(error.asJsonObject().toString(), exitInfo.getPid());
            }
            log.debug("AEI: inspected [" + applicationExitInfoList.size() + "] records: new[" + recordsVisited.get() + "] existing [" + recordsSkipped.get() + "] dropped[" + recordsDropped.get() + "]");
            AEISessionMapper.AEISessionMeta model = new AEISessionMapper.AEISessionMeta(AgentConfiguration.getInstance().getSessionID(), Harvest.getHarvestConfiguration().getDataToken().getAgentId());
            this.sessionMapper.put(this.getCurrentProcessId(), model);
            this.sessionMapper.flush();
            this.reconcileMetadata(applicationExitInfoList);
        } else {
            log.warn("ApplicationExitMonitor: exit info reporting was enabled, but not supported by the current OS");
            StatsEngine.SUPPORTABILITY.inc("Supportability/AgentHealth/ApplicationExitInfo/unsupportedOS/" + Build.VERSION.SDK_INT);
        }
    }

    @RequiresApi(api=30)
    @NonNull
    protected HashMap<String, Object> getEventAttributesForAEI(ApplicationExitInfo exitInfo, AEISessionMapper.AEISessionMeta sessionMeta, String traceReport) throws UnsupportedEncodingException {
        HashMap<String, Object> eventAttributes = new HashMap<String, Object>();
        eventAttributes.put("exitTimestamp", exitInfo.getTimestamp());
        eventAttributes.put("reason", exitInfo.getReason());
        eventAttributes.put("importance", exitInfo.getImportance());
        eventAttributes.put("importanceAsString", this.getImportanceAsString(exitInfo.getImportance()));
        eventAttributes.put("description", this.toValidAttributeValue(exitInfo.getDescription()));
        eventAttributes.put("processName", this.toValidAttributeValue(exitInfo.getProcessName()));
        if (sessionMeta != null) {
            eventAttributes.put("sessionId", sessionMeta.sessionId);
        }
        eventAttributes.put("appExitId", UUID.randomUUID().toString());
        eventAttributes.put("processId", exitInfo.getPid());
        eventAttributes.put("eventType", "MobileApplicationExit");
        switch (exitInfo.getImportance()) {
            case 100: 
            case 125: 
            case 200: 
            case 230: 
            case 325: {
                eventAttributes.put("appState", "foreground");
                break;
            }
            default: {
                eventAttributes.put("appState", "background");
            }
        }
        if (exitInfo.getReason() == 6) {
            AEITrace aeiTrace = new AEITrace();
            aeiTrace.decomposeFromSystemTrace(traceReport);
            eventAttributes.put("threads", URLEncoder.encode(aeiTrace.toString(), StandardCharsets.UTF_8.toString()));
        }
        return eventAttributes;
    }

    @RequiresApi(api=30)
    public Set<Integer> currentPidSet(List<ApplicationExitInfo> applicationExitInfoList) {
        return applicationExitInfoList.stream().mapToInt(applicationExitInfo -> applicationExitInfo.getPid()).boxed().collect(Collectors.toSet());
    }

    @RequiresApi(api=30)
    void reconcileMetadata(List<ApplicationExitInfo> applicationExitInfoList) {
        List<File> artifacts = this.getArtifacts();
        Pattern regexp = Pattern.compile(String.format(Locale.getDefault(), ARTIFACT_NAME, "(\\d+)"));
        Set<Integer> currentPids = this.currentPidSet(applicationExitInfoList);
        artifacts.forEach(aeiArtifact -> {
            int pid;
            Matcher matcher = regexp.matcher(aeiArtifact.getName());
            if (matcher.matches() && !currentPids.contains(pid = Integer.valueOf(matcher.group(1)).intValue())) {
                aeiArtifact.delete();
                this.sessionMapper.erase(pid);
            }
        });
        this.sessionMapper.flush();
    }

    public void resetSessionMap() {
        this.sessionMapper.delete();
    }

    protected String toValidAttributeValue(String attributeValue) {
        return null == attributeValue ? "null" : attributeValue.substring(0, Math.min(attributeValue.length(), 4095));
    }

    protected String getReasonAsString(int reason) {
        return REASON_MAP.getOrDefault(reason, String.valueOf(reason));
    }

    protected String getImportanceAsString(int importance) {
        return IMPORTANCE_MAP.getOrDefault(importance, String.valueOf(importance));
    }

    List<File> getArtifacts() {
        String regexp = String.format(Locale.getDefault(), ARTIFACT_NAME, "\\d+");
        return Streams.list(this.reportsDir).filter(file -> file.isFile() && file.getName().matches(regexp)).collect(Collectors.toList());
    }

    static {
        REASON_MAP.put(0, "Unknown");
        REASON_MAP.put(1, "Exit self");
        REASON_MAP.put(2, "Signaled");
        REASON_MAP.put(3, "Low memory");
        REASON_MAP.put(4, "Crash");
        REASON_MAP.put(5, "Native crash");
        REASON_MAP.put(6, "ANR");
        REASON_MAP.put(7, "Initialization failure");
        REASON_MAP.put(8, "Permission change");
        REASON_MAP.put(9, "Excessive resource usage");
        REASON_MAP.put(10, "User requested");
        REASON_MAP.put(11, "User stopped");
        REASON_MAP.put(12, "Dependency died");
        REASON_MAP.put(13, "Other");
        REASON_MAP.put(14, "Freezer");
        REASON_MAP.put(15, "Package state changed");
        REASON_MAP.put(16, "Package updated");
        IMPORTANCE_MAP.put(100, "Foreground");
        IMPORTANCE_MAP.put(125, "Foreground service");
        IMPORTANCE_MAP.put(325, "Top sleeping");
        IMPORTANCE_MAP.put(200, "Visible");
        IMPORTANCE_MAP.put(230, "Perceptible");
        IMPORTANCE_MAP.put(350, "Can't save state");
        IMPORTANCE_MAP.put(300, "Service");
        IMPORTANCE_MAP.put(400, "Cached");
        IMPORTANCE_MAP.put(1000, "Gone");
    }
}

