/*
 * Decompiled with CFR 0.152.
 */
package com.newrelic.agent;

import com.newrelic.agent.Agent;
import com.newrelic.agent.ExtendedTransactionListener;
import com.newrelic.agent.HarvestListener;
import com.newrelic.agent.Transaction;
import com.newrelic.agent.TransactionData;
import com.newrelic.agent.TransactionListener;
import com.newrelic.agent.TransactionStatsListener;
import com.newrelic.agent.deps.com.google.common.collect.MapMaker;
import com.newrelic.agent.service.AbstractService;
import com.newrelic.agent.service.ServiceFactory;
import com.newrelic.agent.stats.StatsEngine;
import com.newrelic.agent.stats.StatsService;
import com.newrelic.agent.stats.StatsWorks;
import com.newrelic.agent.stats.TransactionStats;
import com.newrelic.agent.transaction.MergeStatsEngineResolvingScope;
import com.newrelic.agent.util.DefaultThreadFactory;
import com.newrelic.agent.util.TimeConversion;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;

public class TransactionService
extends AbstractService {
    private static final String TRANSACTION_SERVICE_PROCESSOR_THREAD_NAME = "New Relic Transaction Service Processor";
    private final List<TransactionListener> transactionListeners = new CopyOnWriteArrayList<TransactionListener>();
    private final List<ExtendedTransactionListener> extendedTransactionListeners = new CopyOnWriteArrayList<ExtendedTransactionListener>();
    private final List<TransactionStatsListener> transactionStatsListeners = new CopyOnWriteArrayList<TransactionStatsListener>();
    private final ScheduledExecutorService scheduler;
    private final ConcurrentMap<Transaction, String> updateQueue;
    private final String placeholder = "placeholder";
    private AtomicLong txStartedThisHarvest = new AtomicLong(0L);
    private AtomicLong txFinishedThisHarvest = new AtomicLong(0L);
    private AtomicLong txCancelledThisHarvest = new AtomicLong(0L);

    public TransactionService() {
        this(1, 5L, 30L, TimeUnit.SECONDS);
    }

    public TransactionService(int numMaintenanceThreads, long initialDelay, long delay, TimeUnit timeUnit) {
        super(TransactionService.class.getSimpleName());
        long initialDelayMilli = TimeConversion.convertToMilliWithLowerBound(initialDelay, timeUnit, 1000L);
        long delayMilli = TimeConversion.convertToMilliWithLowerBound(delay, timeUnit, 1000L);
        this.updateQueue = new MapMaker().concurrencyLevel(16).makeMap();
        this.scheduler = Executors.newScheduledThreadPool(numMaintenanceThreads, new DefaultThreadFactory(TRANSACTION_SERVICE_PROCESSOR_THREAD_NAME, true));
        this.scheduler.scheduleWithFixedDelay(new Runnable(){

            @Override
            public void run() {
                TransactionService.this.processQueue();
            }
        }, initialDelayMilli, delayMilli, TimeUnit.MILLISECONDS);
    }

    public void processQueue() {
        try {
            int transactionCount = 0;
            Iterator txi = this.updateQueue.keySet().iterator();
            while (txi.hasNext()) {
                ((Transaction)txi.next()).cleanUp();
                ++transactionCount;
            }
            this.getLogger().finer("Transaction service processed " + transactionCount + " transactions");
        }
        catch (Throwable t) {
            this.getLogger().log(Level.WARNING, t, "Exception processing async update queue.");
        }
    }

    public void transactionStarted(Transaction transaction) {
        if (transaction != null) {
            this.updateQueue.put(transaction, "placeholder");
            this.txStartedThisHarvest.incrementAndGet();
            if (transaction.getDispatcher() != null) {
                for (ExtendedTransactionListener listener : this.extendedTransactionListeners) {
                    listener.dispatcherTransactionStarted(transaction);
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void transactionFinished(TransactionData transactionData, TransactionStats transactionStats) {
        try {
            this.doProcessTransaction(transactionData, transactionStats);
            this.txFinishedThisHarvest.incrementAndGet();
        }
        catch (Exception e) {
            this.getLogger().log(Level.WARNING, e, "Error recording transaction \"{0}\"", transactionData.getBlameMetricName());
        }
        finally {
            this.updateQueue.remove(transactionData.getTransaction());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void transactionCancelled(Transaction transaction) {
        try {
            this.txCancelledThisHarvest.incrementAndGet();
            if (transaction.getDispatcher() != null) {
                for (ExtendedTransactionListener listener : this.extendedTransactionListeners) {
                    listener.dispatcherTransactionCancelled(transaction);
                }
            }
        }
        finally {
            this.updateQueue.remove(transaction);
        }
    }

    private void doProcessTransaction(TransactionData transactionData, TransactionStats transactionStats) {
        if (!ServiceFactory.getServiceManager().isStarted() || !ServiceFactory.getAgent().isEnabled()) {
            return;
        }
        if (Agent.isDebugEnabled()) {
            this.getLogger().finer("Recording metrics for " + transactionData);
        }
        boolean sizeLimitExceeded = transactionData.getAgentAttributes().get("size_limit") != null;
        transactionStats.getUnscopedStats().getStats("Supportability/TransactionSize").recordDataPoint(transactionData.getTransactionSize());
        if (sizeLimitExceeded) {
            transactionStats.getUnscopedStats().getStats("Supportability/TransactionSizeClamp").incrementCallCount();
        }
        if (transactionData.getDispatcher() != null) {
            for (TransactionListener transactionListener : this.transactionListeners) {
                transactionListener.dispatcherTransactionFinished(transactionData, transactionStats);
            }
            for (ExtendedTransactionListener extendedTransactionListener : this.extendedTransactionListeners) {
                extendedTransactionListener.dispatcherTransactionFinished(transactionData, transactionStats);
            }
        } else if (Agent.isDebugEnabled()) {
            this.getLogger().finer("Skipping transaction trace for " + transactionData);
        }
        StatsService statsService = ServiceFactory.getStatsService();
        MergeStatsEngineResolvingScope mergeStatsEngineResolvingScope = new MergeStatsEngineResolvingScope(transactionData.getBlameMetricName(), transactionData.getApplicationName(), transactionStats);
        statsService.doStatsWork(mergeStatsEngineResolvingScope);
        if (transactionData.getDispatcher() != null) {
            for (TransactionStatsListener listener : this.transactionStatsListeners) {
                listener.dispatcherTransactionStatsFinished(transactionData, transactionStats);
            }
        }
    }

    @Override
    protected void doStart() {
        this.getLogger().finer("Transaction service starting");
        ServiceFactory.getHarvestService().addHarvestListener(new HarvestListener(){
            private volatile long txStarted = 0L;
            private volatile long txFinished = 0L;
            private volatile long txCancelled = 0L;

            @Override
            public void beforeHarvest(String appName, StatsEngine statsEngine) {
            }

            @Override
            public void afterHarvest(String appName) {
                long started = TransactionService.this.txStartedThisHarvest.getAndSet(0L);
                long finished = TransactionService.this.txFinishedThisHarvest.getAndSet(0L);
                long cancelled = TransactionService.this.txCancelledThisHarvest.getAndSet(0L);
                this.txStarted += started;
                this.txFinished += finished;
                this.txCancelled += cancelled;
                TransactionService.this.recordTransactionSupportabilityMetrics(started, finished, cancelled);
                Agent.LOG.log(Level.FINE, "TransactionService: harvest: s/f/c {0}/{1}/{2}, total {3}/{4}/{5}, queue {6}", started, finished, cancelled, this.txStarted, this.txFinished, this.txCancelled, TransactionService.this.updateQueue.size());
            }
        });
    }

    private void recordTransactionSupportabilityMetrics(long started, long finished, long cancelled) {
        StatsService statsService = ServiceFactory.getStatsService();
        statsService.doStatsWork(StatsWorks.getRecordMetricWork("Supportability/Transaction/Harvest/StartedCount", started));
        statsService.doStatsWork(StatsWorks.getRecordMetricWork("Supportability/Transaction/Harvest/FinishedCount", finished));
        statsService.doStatsWork(StatsWorks.getRecordMetricWork("Supportability/Transaction/Harvest/CancelledCount", cancelled));
        statsService.doStatsWork(StatsWorks.getIncrementCounterWork("Supportability/Transaction/StartedCount", (int)started));
        statsService.doStatsWork(StatsWorks.getIncrementCounterWork("Supportability/Transaction/FinishedCount", (int)finished));
        statsService.doStatsWork(StatsWorks.getIncrementCounterWork("Supportability/Transaction/CancelledCount", (int)cancelled));
    }

    @Override
    protected void doStop() {
        this.getLogger().finer("Transaction service stopping");
        this.transactionListeners.clear();
        this.extendedTransactionListeners.clear();
        this.transactionStatsListeners.clear();
        this.updateQueue.clear();
        this.shutdownQueue();
    }

    private void shutdownQueue() {
        this.getLogger().finer("Attempting graceful shutdown of transaction service");
        this.scheduler.shutdown();
        try {
            if (!this.scheduler.awaitTermination(30L, TimeUnit.SECONDS)) {
                this.getLogger().finer("Graceful shutdown timed out, attempting forceful shutdown of transaction service");
                this.scheduler.shutdownNow();
                if (!this.scheduler.awaitTermination(15L, TimeUnit.SECONDS)) {
                    this.getLogger().finer("Forceful shutdown timed out");
                }
            }
        }
        catch (InterruptedException e) {
            this.scheduler.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    public void addTransactionListener(TransactionListener listener) {
        this.transactionListeners.add(listener);
    }

    public void removeTransactionListener(TransactionListener listener) {
        this.transactionListeners.remove(listener);
    }

    public void addTransactionListener(ExtendedTransactionListener listener) {
        this.extendedTransactionListeners.add(listener);
    }

    public void removeTransactionListener(ExtendedTransactionListener listener) {
        this.extendedTransactionListeners.remove(listener);
    }

    public void addTransactionStatsListener(TransactionStatsListener listener) {
        this.transactionStatsListeners.add(listener);
    }

    public void removeTransactionStatsListener(TransactionStatsListener listener) {
        this.transactionStatsListeners.remove(listener);
    }

    public int getTransactionsInProgress() {
        return this.updateQueue.size();
    }

    @Override
    public boolean isEnabled() {
        return true;
    }

    public Transaction getTransaction(boolean createIfNotExists) {
        return Transaction.getTransaction(createIfNotExists);
    }
}

