/*
 * Decompiled with CFR 0.152.
 */
package com.newrelic.labs;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.newrelic.labs.FlushCallback;
import com.newrelic.labs.LogEntry;
import com.newrelic.labs.LowercaseKeyMap;
import com.newrelic.labs.NRBufferWithFifoEviction;
import com.newrelic.telemetry.Attributes;
import com.newrelic.telemetry.OkHttpPoster;
import com.newrelic.telemetry.TelemetryClient;
import com.newrelic.telemetry.logs.Log;
import com.newrelic.telemetry.logs.LogBatch;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.zip.GZIPOutputStream;

public class LogForwarder {
    private final NRBufferWithFifoEviction<LogEntry> logQueue;
    private final String apiKey;
    private final String apiURL;
    private final ObjectMapper objectMapper;
    private final long maxMessageSize;
    private final int maxRetries;
    private final long timeout;
    private final TelemetryClient telemetryClient;

    public LogForwarder(String apiKey, String apiURL, long maxMessageSize, NRBufferWithFifoEviction<LogEntry> queue, int maxRetries, long timeout, int connPoolSize) {
        this.apiKey = apiKey;
        this.apiURL = apiURL;
        this.maxMessageSize = maxMessageSize;
        this.logQueue = queue;
        this.maxRetries = maxRetries;
        this.timeout = timeout;
        this.telemetryClient = TelemetryClient.create(() -> new OkHttpPoster(Duration.of(this.timeout, ChronoUnit.MILLIS)), (String)apiKey);
        this.objectMapper = new ObjectMapper();
    }

    public void close(boolean mergeCustomFields, Map<String, Object> customFields) {
        ArrayList<LogEntry> remainingLogs = new ArrayList<LogEntry>();
        int drained = this.logQueue.drainTo((Collection<LogEntry>)remainingLogs, Integer.MAX_VALUE);
        if (!remainingLogs.isEmpty()) {
            System.out.println("Flushing remaining " + remainingLogs.size() + " log events to New Relic...");
            this.flushAsync(remainingLogs, mergeCustomFields, customFields, new FlushCallback(){

                @Override
                public void onFailure(List<Map<String, Object>> failedLogEvents) {
                    System.err.println("Failed to flush remaining log events to New Relic.");
                }

                @Override
                public void onSuccess() {
                    System.out.println("No remaining log events to flush.");
                }
            });
        }
        this.shutdown();
    }

    public LogEntry convertToLogEntry(Map<String, Object> logEvent) {
        try {
            return (LogEntry)this.objectMapper.convertValue(logEvent, LogEntry.class);
        }
        catch (IllegalArgumentException e) {
            System.err.println("Failed to convert log event to LogEntry: " + e.getMessage());
            return null;
        }
    }

    private Map<String, Object> convertToLogEvent(LogEntry entry, boolean mergeCustomFields, Map<String, Object> customFields) {
        Map logEvent = (Map)this.objectMapper.convertValue((Object)entry, LowercaseKeyMap.class);
        try {
            InetAddress localhost = InetAddress.getLocalHost();
            String hostname = localhost != null ? localhost.getHostName() : "unknown";
            logEvent.put("hostname", hostname);
        }
        catch (UnknownHostException e) {
            System.err.println("Error resolving local host: " + e.getMessage());
        }
        logEvent.put("logtype", entry.getLogType());
        logEvent.put("timestamp", entry.getTimestamp());
        logEvent.put("applicationName", entry.getApplicationName());
        logEvent.put("name", entry.getName());
        logEvent.put("source", "NRBatchingAppender");
        if (customFields != null) {
            if (mergeCustomFields) {
                for (Map.Entry<String, Object> field : customFields.entrySet()) {
                    logEvent.put(field.getKey(), field.getValue());
                }
            } else {
                logEvent.put("custom", customFields);
            }
        }
        return logEvent;
    }

    private List<Map<String, Object>> convertToLogEvents(List<LogEntry> logEntries, boolean mergeCustomFields, Map<String, Object> customFields) {
        ArrayList<Map<String, Object>> logEvents = new ArrayList<Map<String, Object>>();
        for (LogEntry entry : logEntries) {
            logEvents.add(this.convertToLogEvent(entry, mergeCustomFields, customFields));
        }
        return logEvents;
    }

    public void flushAsync(List<LogEntry> logEntries, boolean mergeCustomFields, Map<String, Object> customFields, FlushCallback callback) {
        List<Map<String, Object>> logEvents = this.convertToLogEvents(logEntries, mergeCustomFields, customFields);
        if ((long)logEvents.size() > this.maxMessageSize) {
            try {
                this.splitAndSendLogsAsync(logEntries, mergeCustomFields, customFields, callback);
            }
            catch (IOException e) {
                e.printStackTrace();
            }
        } else {
            this.sendLogsAsync(logEntries, callback, mergeCustomFields, customFields);
        }
    }

    private String getHostname() {
        try {
            InetAddress localhost = InetAddress.getLocalHost();
            return localhost != null ? localhost.getHostName() : "unknown";
        }
        catch (UnknownHostException e) {
            System.err.println("Error resolving local host: " + e.getMessage());
            return "unknown";
        }
    }

    private byte[] gzipCompress(String input) throws IOException {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        try (GZIPOutputStream gzipOS = new GZIPOutputStream(bos);){
            gzipOS.write(input.getBytes());
        }
        return bos.toByteArray();
    }

    public boolean isInitialized() {
        return this.apiKey != null && this.apiURL != null;
    }

    public void onFaolit(List<LogEntry> logEntries, boolean mergeCustomFields, Map<String, Object> customFields, FlushCallback callback) {
        List<Map<String, Object>> logEvents = this.convertToLogEvents(logEntries, mergeCustomFields, customFields);
        if ((long)logEvents.size() > this.maxMessageSize) {
            try {
                this.splitAndSendLogsAsync(logEntries, mergeCustomFields, customFields, callback);
            }
            catch (IOException e) {
                e.printStackTrace();
            }
        } else {
            this.sendLogsAsync(logEntries, callback, mergeCustomFields, customFields);
        }
    }

    private void requeueLogs(List<Map<String, Object>> logEvents) {
        for (Map<String, Object> logEvent : logEvents) {
            try {
                LogEntry logEntry = (LogEntry)this.objectMapper.convertValue(logEvent, LogEntry.class);
                boolean added = this.logQueue.add(logEntry);
                if (added) continue;
                System.err.println("Failed to add log entry to the queue, possibly due to size constraints.");
            }
            catch (IllegalArgumentException e) {
                System.err.println("Failed to convert log event to LogEntry: " + logEvent);
            }
        }
        System.err.println("Network issue - NewRelicBatchingAppenderhas re-queued " + logEvents.size() + " entries : queue size " + this.logQueue.size());
    }

    private void sendLogsAsync(List<LogEntry> logEntries, FlushCallback callback, boolean mergeCustomFields, Map<String, Object> customFields) {
        try {
            List logs = logEntries.stream().map(entry -> Log.builder().message(entry.getMessage()).timestamp(entry.getTimestamp()).attributes(new Attributes().put("applicationName", entry.getApplicationName()).put("logtype", entry.getLogType()).put("name", entry.getName()).put("hostname", this.getHostname()).put("source", "NRBatchingAppender")).build()).collect(Collectors.toList());
            Attributes commonAttributes = new Attributes();
            if (customFields != null) {
                if (mergeCustomFields) {
                    for (Map.Entry<String, Object> field : customFields.entrySet()) {
                        commonAttributes.put(field.getKey(), (String)field.getValue());
                    }
                } else {
                    for (Map.Entry<String, Object> field : customFields.entrySet()) {
                        commonAttributes.put("custom." + field.getKey(), (String)field.getValue());
                    }
                }
            }
            commonAttributes.put("source", "NRBatchingAppender");
            commonAttributes.put("version", "1.0.7");
            LogBatch logBatch = new LogBatch(logs, commonAttributes);
            this.telemetryClient.sendBatch(logBatch);
            callback.onSuccess();
        }
        catch (Exception e) {
            System.err.println("Failed to send logs asynchronously: " + e.getMessage());
            callback.onFailure(this.convertToLogEvents(logEntries, mergeCustomFields, customFields));
        }
    }

    public void shutdown() {
        this.telemetryClient.shutdown();
        System.out.println("Shutting down telemetry client ");
    }

    private void splitAndSendLogsAsync(List<LogEntry> logEntries, boolean mergeCustomFields, Map<String, Object> customFields, FlushCallback callback) throws IOException {
        ArrayList<LogEntry> subBatch = new ArrayList<LogEntry>();
        int currentSize = 0;
        for (LogEntry entry : logEntries) {
            subBatch.add(entry);
            String entryJson = this.objectMapper.writeValueAsString(this.convertToLogEvent(entry, mergeCustomFields, customFields));
            int entrySize = this.gzipCompress(entryJson).length;
            if ((long)(currentSize + entrySize) > this.maxMessageSize) {
                this.sendLogsAsync(logEntries, callback, mergeCustomFields, customFields);
                subBatch.clear();
                currentSize = 0;
            }
            currentSize += entrySize;
        }
        if (!subBatch.isEmpty()) {
            this.sendLogsAsync(logEntries, callback, mergeCustomFields, customFields);
        }
    }
}

