/*
 * Decompiled with CFR 0.152.
 */
package com.newrelic.labs;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.newrelic.labs.FlushCallback;
import com.newrelic.labs.LogEntry;
import com.newrelic.labs.LowercaseKeyMap;
import com.newrelic.labs.NRBufferWithFifoEviction;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.zip.GZIPOutputStream;
import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.ConnectionPool;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;

public class LogForwarder {
    private final NRBufferWithFifoEviction<LogEntry> logQueue;
    private final String apiKey;
    private final String apiURL;
    private final OkHttpClient client;
    private final ObjectMapper objectMapper;
    private final long maxMessageSize;
    private final int maxRetries;
    private final int connPoolSize;
    private final long timeout;

    public LogForwarder(String apiKey, String apiURL, long maxMessageSize, NRBufferWithFifoEviction<LogEntry> queue, int maxRetries, long timeout, int connPoolSize) {
        this.apiKey = apiKey;
        this.apiURL = apiURL;
        this.maxMessageSize = maxMessageSize;
        this.logQueue = queue;
        this.maxRetries = maxRetries;
        this.timeout = timeout;
        this.connPoolSize = connPoolSize;
        ConnectionPool connectionPool = new ConnectionPool(connPoolSize, 5L, TimeUnit.MINUTES);
        this.client = new OkHttpClient.Builder().connectTimeout(timeout, TimeUnit.MILLISECONDS).connectionPool(connectionPool).build();
        this.objectMapper = new ObjectMapper();
    }

    public boolean isInitialized() {
        return this.apiKey != null && this.apiURL != null;
    }

    public boolean flush(List<LogEntry> logEntries, boolean mergeCustomFields, Map<String, Object> customFields) {
        boolean bStatus = false;
        try {
            List<Map<String, Object>> logEvents = this.convertToLogEvents(logEntries, mergeCustomFields, customFields);
            String jsonPayload = this.objectMapper.writeValueAsString(logEvents);
            byte[] compressedPayload = this.gzipCompress(jsonPayload);
            bStatus = (long)compressedPayload.length > this.maxMessageSize ? this.splitAndSendLogs(logEntries, mergeCustomFields, customFields) : this.sendLogs(logEvents);
        }
        catch (IOException e) {
            System.err.println("Error during log forwarding: " + e.getMessage());
            bStatus = false;
        }
        return bStatus;
    }

    public void flushAsync(List<LogEntry> logEntries, boolean mergeCustomFields, Map<String, Object> customFields, FlushCallback callback) {
        List<Map<String, Object>> logEvents = this.convertToLogEvents(logEntries, mergeCustomFields, customFields);
        if ((long)logEvents.size() > this.maxMessageSize) {
            try {
                this.splitAndSendLogsAsync(logEntries, mergeCustomFields, customFields, callback);
            }
            catch (IOException e) {
                e.printStackTrace();
            }
        } else {
            this.sendLogsAsync(logEvents, callback);
        }
    }

    private boolean splitAndSendLogs(List<LogEntry> logEntries, boolean mergeCustomFields, Map<String, Object> customFields) throws IOException {
        ArrayList<LogEntry> subBatch = new ArrayList<LogEntry>();
        int currentSize = 0;
        boolean bStatus = false;
        for (LogEntry entry : logEntries) {
            subBatch.add(entry);
            String entryJson = this.objectMapper.writeValueAsString(this.convertToLogEvent(entry, mergeCustomFields, customFields));
            int entrySize = this.gzipCompress(entryJson).length;
            if ((long)(currentSize + entrySize) > this.maxMessageSize) {
                bStatus = this.sendLogs(this.convertToLogEvents(subBatch, mergeCustomFields, customFields));
                subBatch.clear();
                currentSize = 0;
            }
            currentSize += entrySize;
        }
        if (!subBatch.isEmpty()) {
            bStatus = this.sendLogs(this.convertToLogEvents(subBatch, mergeCustomFields, customFields));
        }
        return bStatus;
    }

    private void splitAndSendLogsAsync(List<LogEntry> logEntries, boolean mergeCustomFields, Map<String, Object> customFields, FlushCallback callback) throws IOException {
        ArrayList<LogEntry> subBatch = new ArrayList<LogEntry>();
        int currentSize = 0;
        for (LogEntry entry : logEntries) {
            subBatch.add(entry);
            String entryJson = this.objectMapper.writeValueAsString(this.convertToLogEvent(entry, mergeCustomFields, customFields));
            int entrySize = this.gzipCompress(entryJson).length;
            if ((long)(currentSize + entrySize) > this.maxMessageSize) {
                this.sendLogsAsync(this.convertToLogEvents(subBatch, mergeCustomFields, customFields), callback);
                subBatch.clear();
                currentSize = 0;
            }
            currentSize += entrySize;
        }
        if (!subBatch.isEmpty()) {
            this.sendLogsAsync(this.convertToLogEvents(subBatch, mergeCustomFields, customFields), callback);
        }
    }

    private List<Map<String, Object>> convertToLogEvents(List<LogEntry> logEntries, boolean mergeCustomFields, Map<String, Object> customFields) {
        ArrayList<Map<String, Object>> logEvents = new ArrayList<Map<String, Object>>();
        for (LogEntry entry : logEntries) {
            logEvents.add(this.convertToLogEvent(entry, mergeCustomFields, customFields));
        }
        return logEvents;
    }

    private Map<String, Object> convertToLogEvent(LogEntry entry, boolean mergeCustomFields, Map<String, Object> customFields) {
        Map logEvent = (Map)this.objectMapper.convertValue((Object)entry, LowercaseKeyMap.class);
        try {
            InetAddress localhost = InetAddress.getLocalHost();
            String hostname = localhost != null ? localhost.getHostName() : "unknown";
            logEvent.put("hostname", hostname);
        }
        catch (UnknownHostException e) {
            System.err.println("Error resolving local host: " + e.getMessage());
        }
        logEvent.put("logtype", entry.getLogType());
        logEvent.put("timestamp", entry.getTimestamp());
        logEvent.put("applicationName", entry.getApplicationName());
        logEvent.put("name", entry.getName());
        logEvent.put("source", "NRBatchingAppender");
        logEvent.put("version", "1.1.10");
        if (customFields != null) {
            if (mergeCustomFields) {
                for (Map.Entry<String, Object> field : customFields.entrySet()) {
                    logEvent.put(field.getKey(), field.getValue());
                }
            } else {
                logEvent.put("custom", customFields);
            }
        }
        return logEvent;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean sendLogs(List<Map<String, Object>> logEvents) throws IOException {
        String jsonPayload = this.objectMapper.writeValueAsString(logEvents);
        byte[] compressedPayload = this.gzipCompress(jsonPayload);
        MediaType mediaType = MediaType.parse((String)"application/json");
        RequestBody requestBody = RequestBody.create((byte[])compressedPayload, (MediaType)mediaType);
        Request request = new Request.Builder().url(this.apiURL).post(requestBody).addHeader("X-License-Key", this.apiKey).addHeader("Content-Type", "application/json").addHeader("Content-Encoding", "gzip").build();
        try (Response response = this.client.newCall(request).execute();){
            if (response.isSuccessful()) return true;
            System.err.println("Failed to send logs to New Relic: " + response.code() + " - " + response.message());
            System.err.println("Response body: " + response.body().string());
            this.requeueLogs(logEvents);
            boolean bl = false;
            return bl;
        }
        catch (IOException e) {
            System.err.println("Error during log forwarding: " + e.getMessage());
            this.requeueLogs(logEvents);
            return false;
        }
    }

    private void requeueLogs(List<Map<String, Object>> logEvents) {
        for (Map<String, Object> logEvent : logEvents) {
            try {
                LogEntry logEntry = (LogEntry)this.objectMapper.convertValue(logEvent, LogEntry.class);
                boolean added = this.logQueue.add(logEntry);
                if (added) continue;
                System.err.println("Failed to add log entry to the queue, possibly due to size constraints.");
            }
            catch (IllegalArgumentException e) {
                System.err.println("Failed to convert log event to LogEntry: " + logEvent);
            }
        }
        System.err.println("Network issue - NewRelicBatchingAppenderhas re-queued " + logEvents.size() + " entries : queue size " + this.logQueue.size());
    }

    private byte[] gzipCompress(String input) throws IOException {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        try (GZIPOutputStream gzipOS = new GZIPOutputStream(bos);){
            gzipOS.write(input.getBytes());
        }
        return bos.toByteArray();
    }

    public void close(boolean mergeCustomFields, Map<String, Object> customFields) {
        ArrayList<LogEntry> remainingLogs = new ArrayList<LogEntry>();
        int drained = this.logQueue.drainTo((Collection<LogEntry>)remainingLogs, Integer.MAX_VALUE);
        if (!remainingLogs.isEmpty()) {
            System.out.println("Flushing remaining " + remainingLogs.size() + " log events to New Relic...");
            boolean success = this.flush(remainingLogs, mergeCustomFields, customFields);
            if (!success) {
                System.err.println("Failed to flush remaining log events to New Relic.");
            }
        } else {
            System.out.println("No remaining log events to flush.");
        }
    }

    private void sendLogsAsync(final List<Map<String, Object>> logEvents, final FlushCallback callback) {
        try {
            String jsonPayload = this.objectMapper.writeValueAsString(logEvents);
            byte[] compressedPayload = this.gzipCompress(jsonPayload);
            MediaType mediaType = MediaType.parse((String)"application/json");
            RequestBody requestBody = RequestBody.create((byte[])compressedPayload, (MediaType)mediaType);
            Request request = new Request.Builder().url(this.apiURL).post(requestBody).addHeader("X-License-Key", this.apiKey).addHeader("Content-Type", "application/json").addHeader("Content-Encoding", "gzip").build();
            this.client.newCall(request).enqueue(new Callback(){

                public void onFailure(Call call, IOException e) {
                    System.err.println("Failed to send logs asynchronously: " + e.getMessage());
                    callback.onFailure(logEvents);
                }

                public void onResponse(Call call, Response response) throws IOException {
                    try {
                        if (!response.isSuccessful()) {
                            System.err.println("Failed to send logs asynchronously: " + response.code() + " - " + response.message());
                            callback.onFailure(logEvents);
                        } else {
                            callback.onSuccess();
                        }
                    }
                    finally {
                        response.close();
                    }
                }
            });
        }
        catch (IOException e) {
            System.err.println("Error during log forwarding: " + e.getMessage());
            callback.onFailure(logEvents);
        }
    }

    public LogEntry convertToLogEntry(Map<String, Object> logEvent) {
        try {
            return (LogEntry)this.objectMapper.convertValue(logEvent, LogEntry.class);
        }
        catch (IllegalArgumentException e) {
            System.err.println("Failed to convert log event to LogEntry: " + e.getMessage());
            return null;
        }
    }
}

