/*
 * Decompiled with CFR 0.152.
 */
package com.newrelic.telemetry;

import com.newrelic.telemetry.Backoff;
import com.newrelic.telemetry.BaseConfig;
import com.newrelic.telemetry.LimitingScheduler;
import com.newrelic.telemetry.LoggingNotificationHandler;
import com.newrelic.telemetry.NotificationHandler;
import com.newrelic.telemetry.Telemetry;
import com.newrelic.telemetry.TelemetryBatch;
import com.newrelic.telemetry.events.EventBatch;
import com.newrelic.telemetry.events.EventBatchSender;
import com.newrelic.telemetry.exceptions.ResponseException;
import com.newrelic.telemetry.exceptions.RetryWithBackoffException;
import com.newrelic.telemetry.exceptions.RetryWithRequestedWaitException;
import com.newrelic.telemetry.exceptions.RetryWithSplitException;
import com.newrelic.telemetry.http.HttpPoster;
import com.newrelic.telemetry.logs.LogBatch;
import com.newrelic.telemetry.logs.LogBatchSender;
import com.newrelic.telemetry.metrics.MetricBatch;
import com.newrelic.telemetry.metrics.MetricBatchSender;
import com.newrelic.telemetry.spans.SpanBatch;
import com.newrelic.telemetry.spans.SpanBatchSender;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TelemetryClient {
    private static final Logger LOG = LoggerFactory.getLogger(TelemetryClient.class);
    private static final int DEFAULT_SHUTDOWN_SECONDS = 3;
    private static final boolean DEFAULT_IS_DAEMON = true;
    private static final int DEFAULT_MAX_TELEMETRY_LIMIT = 1000000;
    private final EventBatchSender eventBatchSender;
    private final MetricBatchSender metricBatchSender;
    private final SpanBatchSender spanBatchSender;
    private final LimitingScheduler scheduler;
    private final int shutdownSeconds;
    private final LogBatchSender logBatchSender;
    private NotificationHandler notificationHandler = new LoggingNotificationHandler(LOG);

    public TelemetryClient(MetricBatchSender metricBatchSender, SpanBatchSender spanBatchSender, EventBatchSender eventBatchSender, LogBatchSender logBatchSender) {
        this(metricBatchSender, spanBatchSender, eventBatchSender, logBatchSender, 3, true);
    }

    public TelemetryClient(MetricBatchSender metricBatchSender, SpanBatchSender spanBatchSender, EventBatchSender eventBatchSender, LogBatchSender logBatchSender, int shutdownSeconds, boolean useDaemonThread) {
        this(metricBatchSender, spanBatchSender, eventBatchSender, logBatchSender, shutdownSeconds, useDaemonThread, 1000000);
    }

    public TelemetryClient(MetricBatchSender metricBatchSender, SpanBatchSender spanBatchSender, EventBatchSender eventBatchSender, LogBatchSender logBatchSender, int shutdownSeconds, boolean useDaemonThread, int maxTelemetryBuffer) {
        this.metricBatchSender = metricBatchSender;
        this.spanBatchSender = spanBatchSender;
        this.eventBatchSender = eventBatchSender;
        this.logBatchSender = logBatchSender;
        this.shutdownSeconds = shutdownSeconds;
        this.scheduler = TelemetryClient.buildScheduler(useDaemonThread, maxTelemetryBuffer);
    }

    public void sendBatch(MetricBatch batch) {
        this.scheduleBatchSend(b -> this.metricBatchSender.sendBatch((MetricBatch)b), batch, 0L, TimeUnit.SECONDS);
    }

    public void sendBatch(SpanBatch batch) {
        this.scheduleBatchSend(b -> this.spanBatchSender.sendBatch((SpanBatch)b), batch, 0L, TimeUnit.SECONDS);
    }

    public void sendBatch(EventBatch batch) {
        this.scheduleBatchSend(b -> this.eventBatchSender.sendBatch((EventBatch)b), batch, 0L, TimeUnit.SECONDS);
    }

    public void sendBatch(LogBatch batch) {
        this.scheduleBatchSend(b -> this.logBatchSender.sendBatch((LogBatch)b), batch, 0L, TimeUnit.SECONDS);
    }

    private void scheduleBatchSend(BatchSender sender, TelemetryBatch<? extends Telemetry> batch, long waitTime, TimeUnit timeUnit) {
        this.scheduleBatchSend(sender, batch, waitTime, timeUnit, Backoff.defaultBackoff());
    }

    private void scheduleBatchSend(BatchSender sender, TelemetryBatch<? extends Telemetry> batch, long waitTime, TimeUnit timeUnit, Backoff backoff) {
        block3: {
            if (this.scheduler.isTerminated()) {
                return;
            }
            try {
                this.scheduler.schedule(batch.size(), () -> this.sendWithErrorHandling(sender, batch, backoff), waitTime, timeUnit);
            }
            catch (RejectedExecutionException e) {
                if (this.notificationHandler == null) break block3;
                this.notificationHandler.noticeError("Problem scheduling batch : ", e, batch);
            }
        }
    }

    private void sendWithErrorHandling(BatchSender batchSender, TelemetryBatch<? extends Telemetry> batch, Backoff backoff) {
        block7: {
            try {
                batchSender.sendBatch(batch);
                LOG.debug("Telemetry - {} - sent", (Object)batch.getClass().getSimpleName());
            }
            catch (RetryWithBackoffException e) {
                this.backoff(batchSender, batch, backoff);
            }
            catch (RetryWithRequestedWaitException e) {
                this.retry(batchSender, batch, e);
            }
            catch (RetryWithSplitException e) {
                this.splitAndSend(batchSender, batch, e);
            }
            catch (ResponseException e) {
                if (this.notificationHandler != null) {
                    this.notificationHandler.noticeError("Received a fatal exception from the New Relic API. Aborting batch send.", e, batch);
                }
            }
            catch (Exception e) {
                if (this.notificationHandler == null) break block7;
                this.notificationHandler.noticeError("Unexpected failure when sending data.", e, batch);
            }
        }
    }

    private <T extends Telemetry> void splitAndSend(BatchSender sender, TelemetryBatch<T> batch, RetryWithSplitException e) {
        if (this.notificationHandler != null) {
            this.notificationHandler.noticeInfo("Batch size too large, splitting and retrying.", e, batch);
        }
        List<TelemetryBatch<TelemetryBatch>> splitBatches = batch.split();
        splitBatches.forEach(metricBatch -> this.scheduleBatchSend(sender, (TelemetryBatch<? extends Telemetry>)metricBatch, 0L, TimeUnit.SECONDS));
    }

    private void retry(BatchSender sender, TelemetryBatch<? extends Telemetry> batch, RetryWithRequestedWaitException e) {
        if (this.notificationHandler != null) {
            this.notificationHandler.noticeInfo(String.format("Batch sending failed. Retrying failed batch after %d %s", new Object[]{e.getWaitTime(), e.getTimeUnit()}), batch);
        }
        this.scheduleBatchSend(sender, batch, e.getWaitTime(), e.getTimeUnit());
    }

    private void backoff(BatchSender sender, TelemetryBatch<? extends Telemetry> batch, Backoff backoff) {
        long newWaitTime = backoff.nextWaitMs();
        if (newWaitTime == -1L) {
            if (this.notificationHandler != null) {
                this.notificationHandler.noticeError(String.format("Max retries exceeded.  Dropping %d pieces of data!", batch.size()), batch);
            }
            return;
        }
        if (this.notificationHandler != null) {
            this.notificationHandler.noticeInfo(String.format("Batch sending failed. Backing off %d %s", new Object[]{newWaitTime, TimeUnit.MILLISECONDS}), batch);
        }
        this.scheduleBatchSend(sender, batch, newWaitTime, TimeUnit.MILLISECONDS, backoff);
    }

    public void shutdown() {
        LOG.info("Shutting down the TelemetryClient background Executor");
        this.scheduler.shutdown();
        try {
            if (!this.scheduler.awaitTermination(this.shutdownSeconds, TimeUnit.SECONDS)) {
                LOG.warn("couldn't shutdown within timeout");
                this.scheduler.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            LOG.error("interrupted graceful shutdown", (Throwable)e);
            Thread.currentThread().interrupt();
        }
    }

    public static TelemetryClient create(Supplier<HttpPoster> httpPosterCreator, String insertApiKey) {
        return TelemetryClient.create(httpPosterCreator, new BaseConfig(insertApiKey));
    }

    public static TelemetryClient create(Supplier<HttpPoster> httpPosterCreator, BaseConfig baseConfig) {
        MetricBatchSender metricBatchSender = MetricBatchSender.create(httpPosterCreator, baseConfig);
        SpanBatchSender spanBatchSender = SpanBatchSender.create(httpPosterCreator, baseConfig);
        EventBatchSender eventBatchSender = EventBatchSender.create(httpPosterCreator, baseConfig);
        LogBatchSender logBatchSender = LogBatchSender.create(httpPosterCreator, baseConfig);
        return new TelemetryClient(metricBatchSender, spanBatchSender, eventBatchSender, logBatchSender);
    }

    private static LimitingScheduler buildScheduler(boolean useDaemonThread, int maxTelemetryBuffer) {
        ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread thread = new Thread(r);
            thread.setDaemon(useDaemonThread);
            return thread;
        });
        return new LimitingScheduler(executor, maxTelemetryBuffer);
    }

    public void withNotificationHandler(NotificationHandler notificationHandler) {
        this.notificationHandler = notificationHandler;
    }

    private static interface BatchSender {
        public void sendBatch(TelemetryBatch<?> var1) throws ResponseException;
    }
}

