package com.nimbusds.openid.connect.provider.jwksetgen;


import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.text.ParseException;
import java.util.LinkedList;
import java.util.List;

import com.thetransactioncompany.json.pretty.PrettyJson;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.Options;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.jwk.*;
import com.nimbusds.jose.jwk.gen.ECKeyGenerator;
import com.nimbusds.jose.jwk.gen.OctetKeyPairGenerator;
import com.nimbusds.jose.jwk.gen.OctetSequenceKeyGenerator;
import com.nimbusds.jose.jwk.gen.RSAKeyGenerator;
import com.nimbusds.jose.util.Base64URL;


/**
 * JWK set generators for Connect2id servers.
 */
public class JWKSetGenerator {
	
	
	/**
	 * The RSA key bit size.
	 */
	public static final int RSA_KEY_BIT_SIZE = 2048;
	
	
	/**
	 * The AES key bit size.
	 */
	public static final int AES_KEY_BIT_SIZE = 128;
	
	
	/**
	 * The HMAC SHA key bit size.
	 */
	public static final int HMAC_SHA_KEY_BIT_SIZE = 256;
	
	
	/**
	 * The subject AES SIV key bit size.
	 */
	public static final int SUBJECT_AES_SIV_KEY_BIT_SIZE = 256;
	
	
	/**
	 * The refresh token AES SIV key bit size.
	 */
	public static final int REFRESH_TOKEN_AES_SIV_KEY_BIT_SIZE = 256;
	
	
	/**
	 * Generates a 2048 bit RSA signing key with the specified key ID.
	 *
	 * @param kid The key ID, {@code null} if not specified.
	 *
	 * @return The RSA key pair.
	 */
	public static RSAKey generateSigningRSAKey(final String kid)
		throws JOSEException {
		
		return new RSAKeyGenerator(RSA_KEY_BIT_SIZE)
			.keyID(kid)
			.keyUse(KeyUse.SIGNATURE)
			.generate();
	}
	
	
	/**
	 * Generates a 2048 bit RSA encryption key with the specified key ID.
	 *
	 * @param kid The key ID, {@code null} if not specified.
	 *
	 * @return The RSA key pair.
	 */
	public static RSAKey generateEncryptionRSAKey(final String kid)
		throws JOSEException {
		
		return new RSAKeyGenerator(RSA_KEY_BIT_SIZE)
			.keyID(kid)
			.keyUse(KeyUse.ENCRYPTION)
			.generate();
	}
	
	
	/**
	 * Generates an EC signing key with the specified curve and key ID.
	 *
	 * @param crv The curve. Must not be {@code null}.
	 * @param kid The key ID, {@code null} if not specified.
	 *
	 * @return The EC key pair.
	 */
	public static ECKey generateSigningECKey(final Curve crv, final String kid)
		throws JOSEException {
		
		return new ECKeyGenerator(crv)
			.keyID(kid)
			.keyUse(KeyUse.SIGNATURE)
			.generate();
	}
	
	
	/**
	 * Generates an EC encryption key with the specified curve and key ID.
	 *
	 * @param crv The curve. Must not be {@code null}.
	 * @param kid The key ID, {@code null} if not specified.
	 *
	 * @return The EC key pair.
	 */
	public static ECKey generateEncryptionECKey(final Curve crv, final String kid)
		throws JOSEException {
		
		return new ECKeyGenerator(crv)
			.keyID(kid)
			.keyUse(KeyUse.ENCRYPTION)
			.generate();
	}
	
	
	/**
	 * Generates an Ed25519 signing key with the specified key ID.
	 *
	 * @param kid The key ID, {@code null} if not specified.
	 *
	 * @return The EC key pair.
	 */
	public static OctetKeyPair generateSigningEd25519Key(final String kid)
		throws JOSEException {
		
		return new OctetKeyPairGenerator(Curve.Ed25519)
			.keyID(kid)
			.keyUse(KeyUse.SIGNATURE)
			.generate();
	}
	
	
	/**
	 * Generates a 128 bit AES encryption key with the specified key ID.
	 *
	 * @param kid The key ID, {@code null} if not specified.
	 *
	 * @return The AES key.
	 */
	public static OctetSequenceKey generateEncryptionAESKey(final String kid)
		throws JOSEException {
		
		return new OctetSequenceKeyGenerator(AES_KEY_BIT_SIZE)
			.keyID(kid)
			.keyUse(KeyUse.ENCRYPTION)
			.generate();
	}
	
	
	/**
	 * Generates a 256 bit HMAC SHA key with key ID "hmac".
	 *
	 * @return The HMAC SHA key.
	 */
	public static OctetSequenceKey generateHMACSHA256Key()
		throws JOSEException {
		
		return new OctetSequenceKeyGenerator(HMAC_SHA_KEY_BIT_SIZE)
			.keyID("hmac")
			.keyUse(KeyUse.SIGNATURE)
			.generate();
	}
	
	
	/**
	 * Generates a 256 bit subject encryption key (intended for AES SIV
	 * mode) with key ID "subject-encrypt".
	 *
	 * @return The subject encryption key.
	 */
	public static OctetSequenceKey generateSubjectEncryptionKey()
		throws JOSEException {
		
		return new OctetSequenceKeyGenerator(SUBJECT_AES_SIV_KEY_BIT_SIZE)
			.keyID("subject-encrypt")
			.keyUse(KeyUse.ENCRYPTION)
			.generate();
	}
	
	
	/**
	 * Generates a 256 bit refresh token encryption key (intended for AES
	 * SIV mode) with key ID "refresh-token-encrypt".
	 *
	 * @return The refresh token encryption key.
	 */
	public static OctetSequenceKey generateRefreshTokenEncryptionKey()
		throws JOSEException {
		
		return new OctetSequenceKeyGenerator(REFRESH_TOKEN_AES_SIV_KEY_BIT_SIZE)
			.keyID("refresh-token-encrypt")
			.keyUse(KeyUse.ENCRYPTION)
			.generate();
	}
	
	
	/**
	 * Generates a new set of rotating signature and encryption keys for a
	 * Connect2id server.
	 *
	 * @param reservedKeyIDs The reserved key IDs, empty if none.
	 * @param withMessage    If {@code true} a message will be printed to
	 *                       standard output.
	 *
	 * @return The generated rotating keys.
	 */
	public List<JWK> generateRotatingKeys(final KeyIDs reservedKeyIDs, final boolean withMessage)
		throws JOSEException {
		
		List<JWK> keys = new LinkedList<>();
		
		KeyIDs keyIDs = new KeyIDs();
		keyIDs.addAll(reservedKeyIDs);
		
		RSAKey rsaKey = generateSigningRSAKey(keyIDs.addRandomUniqueKeyID());
		keys.add(rsaKey);
		if (withMessage) {
			System.out.println("[1] Generated new signing RSA " + RSA_KEY_BIT_SIZE + " bit key with ID " + rsaKey.getKeyID());
		}
		
		ECKey ecKey = generateSigningECKey(Curve.P_256, keyIDs.addRandomUniqueKeyID());
		keys.add(ecKey);
		if (withMessage) {
			System.out.println("[2] Generated new signing EC " + ecKey.getCurve() + " key with ID " + ecKey.getKeyID());
		}
		
		ecKey = generateSigningECKey(Curve.P_384, keyIDs.addRandomUniqueKeyID());
		keys.add(ecKey);
		if (withMessage) {
			System.out.println("[3] Generated new signing EC " + ecKey.getCurve() + " key with ID " + ecKey.getKeyID());
		}
		
		ecKey = generateSigningECKey(Curve.P_521, keyIDs.addRandomUniqueKeyID());
		keys.add(ecKey);
		if (withMessage) {
			System.out.println("[4] Generated new signing EC " + ecKey.getCurve() + " key with ID " + ecKey.getKeyID());
		}
		
		OctetKeyPair okp = generateSigningEd25519Key(keyIDs.addRandomUniqueKeyID());
		keys.add(okp);
		if (withMessage) {
			System.out.println("[5] Generated new signing " + okp.getCurve() + " key with ID " + okp.getKeyID());
		}
		
		rsaKey = generateEncryptionRSAKey(keyIDs.addRandomUniqueKeyID());
		keys.add(rsaKey);
		if (withMessage) {
			System.out.println("[6] Generated new encryption RSA " + RSA_KEY_BIT_SIZE + " bit key with ID " + rsaKey.getKeyID());
		}
		
		ecKey = generateEncryptionECKey(Curve.P_256, keyIDs.addRandomUniqueKeyID());
		keys.add(ecKey);
		if (withMessage) {
			System.out.println("[7] Generated new encryption EC " + ecKey.getCurve() + " key with ID " + ecKey.getKeyID());
		}
		
		ecKey = generateEncryptionECKey(Curve.P_384, keyIDs.addRandomUniqueKeyID());
		keys.add(ecKey);
		if (withMessage) {
			System.out.println("[8] Generated new encryption EC " + ecKey.getCurve() + " key with ID " + ecKey.getKeyID());
		}
		
		ecKey = generateEncryptionECKey(Curve.P_521, keyIDs.addRandomUniqueKeyID());
		keys.add(ecKey);
		if (withMessage) {
			System.out.println("[9] Generated new encryption EC " + ecKey.getCurve() + " key with ID " + ecKey.getKeyID());
		}
		
		OctetSequenceKey secretKey = generateEncryptionAESKey(keyIDs.addRandomUniqueKeyID());
		keys.add(secretKey);
		if (withMessage) {
			System.out.println("[10] Generated new encryption AES " + AES_KEY_BIT_SIZE + " bit key with ID " + secretKey.getKeyID());
		}
		
		return keys;
	}
	
	
	/**
	 * Generates a new set of permanent keys for a Connect2id server.
	 *
	 * @param withMessage If {@code true} a message will be printed to
	 *                    standard output.
	 *
	 * @return The generated keys.
	 */
	public List<JWK> generatePermanentKeys(final boolean withMessage)
		throws JOSEException {
		
		List<JWK> keys = new LinkedList<>();
		
		OctetSequenceKey hmacKey = generateHMACSHA256Key();
		keys.add(hmacKey);
		if (withMessage) {
			System.out.println("[11] Generated new HMAC SHA " + HMAC_SHA_KEY_BIT_SIZE + " bit key with ID " + hmacKey.getKeyID());
		}
		
		OctetSequenceKey subjectKey = generateSubjectEncryptionKey();
		keys.add(subjectKey);
		if (withMessage) {
			System.out.println("[12] Generated new subject encryption AES SIV " + SUBJECT_AES_SIV_KEY_BIT_SIZE + " bit key with ID " + subjectKey.getKeyID());
		}
		
		OctetSequenceKey refreshTokenKey = generateRefreshTokenEncryptionKey();
		keys.add(refreshTokenKey);
		if (withMessage) {
			System.out.println("[13] Generated new refresh token encryption AES SIV " + REFRESH_TOKEN_AES_SIV_KEY_BIT_SIZE + " bit key with ID " + refreshTokenKey.getKeyID());
		}
		
		return keys;
	}
	
	
	/**
	 * Generates a new JWK set for a Connect2id server.
	 *
	 * @param withMessage If {@code true} a message will be printed to
	 *                    standard output.
	 *
	 * @return The JWK set.
	 */
	public JWKSet generate(final boolean withMessage)
		throws JOSEException {
		
		List<JWK> keys = new LinkedList<>();
		keys.addAll(generateRotatingKeys(new KeyIDs(), withMessage));
		keys.addAll(generatePermanentKeys(withMessage));
		return new JWKSet(keys);
	}
	
	
	/**
	 * A generates a new set of signing and encryption keys and prefixes
	 * them to the specified Connect2id server JWK set.
	 *
	 * @param oldJWKSet   The Connect2id server JWK set. Must not be
	 *                    {@code null}.
	 * @param withMessage If {@code true} a message will be printed to std
	 *                    output.
	 *
	 * @return The updated JWK set.
	 */
	public JWKSet generateAndPrefixNewKeys(final JWKSet oldJWKSet, final boolean withMessage)
		throws Exception {
		
		// Prefix so Connect2id server can roll over to new keys
		List<JWK> keys = new LinkedList<>();
		keys.addAll(generateRotatingKeys(new KeyIDs(oldJWKSet), withMessage));
		keys.addAll(oldJWKSet.getKeys());
		
		if (withMessage) {
			System.out.println("[11] Prefixed newly generated keys to existing JWK set");
		}
		
		return new JWKSet(keys);
	}
	
	
	/**
	 * Prints the CLI usage to standard output.
	 */
	static void printUsage() {
		System.out.println("Usage:");
		System.out.println("1) Generate new Connect2id server JWK set: ");
		System.out.println("   java -jar jwkset-gen.jar jwkSet.json");
		System.out.println("2) Generate new set of rotating keys and add to existing Connect2id server JWK set: ");
		System.out.println("   java -jar jwkset-gen.jar oldJWKSet.json newJWKSet.json");
		System.out.println("3) BASE64URL encode the output JWK set: ");
		System.out.println("   java -jar -b64 jwkset-gen.jar oldJWKSet.json newJWKSet.json.b64");
	}
	
	
	/**
	 * Console method for generating a new Connect2id server JWK set, or
	 * updating an existing JWK set with new signing and encryption keys.
	 *
	 * @param args The command line arguments.
	 */
	public static void main(final String[] args) {
		
		String softwareVersion = JWKSetGenerator.class.getPackage().getImplementationVersion();
		
		System.out.println("JWK set generator " + (softwareVersion != null ? "v" + softwareVersion + " " : "") + "for Connect2id server v6.x+");
		
		// create Options object
		Options options = new Options();
		options.addOption("b64", false, "BASE64URL encode the output JWK set");
		
		CommandLineParser parser = new DefaultParser();
		CommandLine commandLine;
		try {
			commandLine = parser.parse( options, args);
		} catch (org.apache.commons.cli.ParseException e) {
			System.err.println( "Command line parse error: " + e.getMessage());
			printUsage();
			return;
		}
		
		final boolean b64Encode = commandLine.hasOption("b64");
		
		final List<String> argList = commandLine.getArgList();
		
		final File oldJWKSetFile;
		final File newJWKSetFile;
		
		if (argList.size() == 2) {
			oldJWKSetFile = new File(argList.get(0));
			newJWKSetFile = new File(argList.get(1));
		} else if (argList.size() == 1) {
			oldJWKSetFile = null;
			newJWKSetFile = new File(argList.get(0));
		} else {
			printUsage();
			return;
		}
		
		JWKSet oldJWKSet = null;
		
		if (oldJWKSetFile != null) {
			try {
				oldJWKSet = JWKSet.load(oldJWKSetFile);
			} catch (IOException | ParseException e) {
				System.err.println("Couldn't read old JWK set file: " + e.getMessage());
				return;
			}
		}
		
		final JWKSet newJWKSet;
		final boolean withMessage = true;
		try {
			if (oldJWKSet == null) {
				newJWKSet = new JWKSetGenerator().generate(withMessage);
			} else {
				newJWKSet = new JWKSetGenerator().generateAndPrefixNewKeys(oldJWKSet, withMessage);
			}
		} catch (Exception e) {
			System.err.println("Couldn't generate JWK key: " + e.getMessage());
			return;
		}
		
		String json = newJWKSet.toJSONObject(false).toJSONString();
		
		String output;
		
		if (b64Encode) {
			output = Base64URL.encode(json).toString();
		} else {
			try {
				output = new PrettyJson(PrettyJson.Style.COMPACT).parseAndFormat(json);
			} catch (ParseException e) {
				System.err.println("Couldn't format JSON: " + e.getMessage());
				return;
			}
		}
		
		try {
			PrintWriter writer = new PrintWriter(newJWKSetFile, "UTF-8");
			writer.write(output);
			writer.write("\n");
			writer.close();
		} catch (IOException e) {
			System.err.println("Couldn't write new JWK set file: " + e.getMessage());
		}
	}
}
