package com.nimbusds.openid.connect.provider.jwksetgen;


import java.util.LinkedList;
import java.util.List;
import java.util.function.Consumer;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jose.jwk.JWK;
import com.nimbusds.jose.jwk.JWKSet;
import com.nimbusds.jose.jwk.KeyUse;
import com.nimbusds.jose.jwk.RSAKey;
import com.nimbusds.jose.jwk.gen.RSAKeyGenerator;


/**
 * JWK set generator for OpenID Connect Federation 1.0 entities.
 */
public class FederationJWKSetGenerator {
	
	
	/**
	 * The RSA key bit size.
	 */
	public static final int RSA_KEY_BIT_SIZE = 2048;
	
	
	/**
	 * Generates a 2048 bit RSA signing key with the specified key ID.
	 *
	 * @param kid The key ID, {@code null} if not specified.
	 *
	 * @return The RSA key pair.
	 */
	public static RSAKey generateSigningRSAKey(final String kid)
		throws JOSEException {
		
		return new RSAKeyGenerator(RSA_KEY_BIT_SIZE)
			.keyID(kid)
			.algorithm(JWSAlgorithm.RS256)
			.keyUse(KeyUse.SIGNATURE)
			.generate();
	}
	
	
	/**
	 * Generates a new set of rotating signature keys for a OpenID Connect
	 * Federation 1.0 entity.
	 *
	 * @param reservedKeyIDs   The reserved key IDs, empty if none.
	 * @param eventMessageSink Optional sink for event messages,
	 *                         {@code null} if not specified.
	 *
	 * @return The generated rotating keys.
	 */
	public List<JWK> generateRotatingKeys(final KeyIDs reservedKeyIDs, final Consumer<String> eventMessageSink)
		throws JOSEException {
		
		List<JWK> keys = new LinkedList<>();
		
		KeyIDs keyIDs = new KeyIDs();
		keyIDs.addAll(reservedKeyIDs);
		
		RSAKey rsaKey = generateSigningRSAKey(keyIDs.addRandomUniqueKeyID());
		keys.add(rsaKey);
		if (eventMessageSink != null) {
			eventMessageSink.accept("Generated new signing RSA " + RSA_KEY_BIT_SIZE + " bit key with ID " + rsaKey.getKeyID());
		}
		
		return keys;
	}
	
	
	/**
	 * Generates a new JWK set for an OpenID Connect Federation 1.0
	 * entity.
	 *
	 * @param eventMessageSink Optional sink for event messages,
	 *                         {@code null} if not specified.
	 *
	 * @return The JWK set.
	 */
	public JWKSet generate(final Consumer<String> eventMessageSink)
		throws JOSEException {
		
		return new JWKSet(generateRotatingKeys(new KeyIDs(), eventMessageSink));
	}
	
	
	/**
	 * A generates a new set of signing keys and prefixes them to the
	 * specified OpenID Connect Federation 1.0 entity JWK set.
	 *
	 * @param oldJWKSet        The OpenID Connect Federation 1.0 entity JWK
	 *                         set. Must not be {@code null}.
	 * @param eventMessageSink Optional sink for event messages,
	 *                         {@code null} if not specified.
	 *
	 * @return The updated JWK set.
	 */
	public JWKSet generateAndPrefixNewKeys(final JWKSet oldJWKSet, final Consumer<String> eventMessageSink)
		throws Exception {
		
		// Prefix so Connect2id server can roll over to new keys
		List<JWK> keys = new LinkedList<>(generateRotatingKeys(new KeyIDs(oldJWKSet), eventMessageSink));
		keys.addAll(oldJWKSet.getKeys());
		
		if (eventMessageSink != null) {
			eventMessageSink.accept("Prefixed newly generated keys to existing federation entity JWK set");
		}
		
		return new JWKSet(keys);
	}
}
