package com.nimbusds.common.ldap;


import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.minidev.json.JSONObject;

import com.unboundid.ldap.sdk.Attribute;
import com.unboundid.ldap.sdk.Entry;
import com.unboundid.util.StaticUtils;

import com.nimbusds.langtag.LangTag;
import com.nimbusds.langtag.LangTagUtil;


/**
 * LDAP attribute mapper. Maps an LDAP entry to a JSON object (and vice versa), 
 * supports attribute renaming and type conversion.
 *
 * <p>Transformation map specification:
 *
 * <ul>
 *     <li>Each map entry specifies the transformation of a single LDAP 
 *         attribute to / from a JSON object member.
 *     <li>The map key represents the JSON object member name.
 *     <li>The map value is a map of the following properties:
 *         <ul>
 *             <li>The "ldapAttr" property specifies the LDAP attribute name.
 *             <li>The optional "ldapType" property specifies the LDAP
 *                 attribute type: "string" or "time", defaults to "string".
 *             <li>The optional "jsonType" property specifies the JSON value
 *                 type: "string", "string-array", "int", "long" or "boolean", 
 *                 defaults to "string".
 *             <li>The optional "langTag" property specifies if the  "string" 
 *                 values may be language-tagged: true or false, defaults to 
 *                 false.
 *         </ul>
 *     </li>
 * </ul>
 * 
 * <p>The "time" LDAP attribute type is defined in 
 * <a href="http://tools.ietf.org/html/rfc4517#section-3.3.13">RFC 4517, 
 * section 3.3.13</a>
 *
 * <p>Example transformation map (as JSON object):
 *
 * <pre>
 * {
 *   "email"   : { "ldapAttr" : "mail" },
 *   "age"     : { "ldapAttr" : "personAge", "jsonType" : "int" },
 *   "name"    : { "ldapAttr" : "cn", "langTag" : true },
 *   "active"  : { "ldapAttr" : "accountActive", "jsonType" : "boolean" },
 *   "updated" : { "ldapAttr" : "updateTime", "ldapType" : "time", "jsonType" : "long" },
 *   ...
 * }
 * </pre>
 *
 * @author Vladimir Dzhuvinov
 */
public class AttributeMapper {
	
	
	/**
	 * Specifies an individual directive for mapping between an LDAP 
	 * attribute and a JSON value. This class is immutable.
	 */
	private static final class Directive {
	
		
		/**
		 * The type of JSON value.
		 */
		private final String jsonType;
		
		
		/**
		 * The name of the LDAP attribute.
		 */
		private final String ldapAttr;
		
		
		/**
		 * The type of the LDAP attribute.
		 */
		private final String ldapType;
		
		
		/**
		 * Specifies if the attribute is language tagged.
		 */
		private final boolean langTagged;
		
		
		/**
		 * Creates a new attribute mapping directive.
		 * 
		 * @param ldapAttr   The name of the LDAP attribute, must not 
		 *                   be {@code null}.
		 * @param ldapType   The type of the LDAP attribute, 
		 *                   {@code null} if not specified.
		 * @param jsonType   The type of JSON value, {@code null} if 
		 *                   not specified.
		 * @param langTagged If {@code true} the values may be language
		 *                   -tagged, else not.
		 */
		public Directive(final String ldapAttr, 
			         final String ldapType, 
				 final String jsonType,
				 final boolean langTagged) {
			
			if (ldapAttr == null)
				throw new IllegalArgumentException("Missing LDAP attribute name");
			
			this.ldapAttr = ldapAttr;
			
			
			if (ldapType == null)
				this.ldapType = "string";
			else
				this.ldapType = ldapType;
			
			if (jsonType == null)
				this.jsonType = "string";
			else
				this.jsonType = jsonType;
			
			this.langTagged = langTagged;
		}
		
		
		/**
		 * Returns the name of the LDAP attribute.
		 * 
		 * @return The LDAP attribute name.
		 */
		public String ldapAttributeName() {
			
			return ldapAttr;
		}
		
		
		/**
		 * Returns the type of the LDAP attribute.
		 * 
		 * @return The LDAP attribute type.
		 */
		public String ldapAttributeType() {
			
			return ldapType;
		}
		
		
		/**
		 * Returns the type of the JSON value.
		 * 
		 * @return The JSON value type.
		 */
		public String jsonValueType() {
			
			return jsonType;
		}
		
		
		/**
		 * Returns {@code true} if the values may be language-tagged.
		 * 
		 * @return {@code true} if the values may be language-tagged.
		 */
		public boolean langTagged() {
			
			return langTagged;
		}
		
		
		/**
		 * Parses an attribute mapping directive from the specified map
		 * representation.
		 * 
		 * <p>Example map representation (as JSON object):
		 * 
		 * <pre>
		 * { "ldapAttr" : "updateTime", "ldapType" : "time", "jsonType" : "long" },
		 * </pre>
		 * 
		 * @param map The map to parse. Must not be {@code null}.
		 * 
		 * @return The attribute mapping directive.
		 * 
		 * @throws ParseException If parsing of the map failed.
		 */
		public static Directive parse(final Map<String,Object> map)
			throws ParseException {

			// Parse LDAP attribute name
			String ldapAttrName = null;

			try {
				ldapAttrName = (String)map.get("ldapAttr");

			} catch (Exception e) {

				throw new ParseException("Invalid LDAP attribute name", 0);
			}

			if (ldapAttrName == null)

			throw new ParseException("Missing LDAP attribute name", 0);


			// Parse LDAP attribute type
			String ldapAttrType = null;

			if (map.containsKey("ldapType")) {

				try {
					ldapAttrType = (String)map.get("ldapType");
				
				} catch (Exception e) {

					throw new ParseException("Invalid LDAP attribute type", 0);
				}
			}

			if (ldapAttrType == null)
				ldapAttrType = "string";

			if (! ldapAttrType.equals("string") &&
			    ! ldapAttrType.equals("time")      )
				throw new ParseException("Invalid attribute LDAP type, must be \"string\" or \"time\"", 0);
			
			
			// Parse JSON value type
			String jsonType = null;
			
			if (map.containsKey("jsonType")) {

				try {
					jsonType = (String)map.get("jsonType");
				
				} catch (Exception e) {

					throw new ParseException("Invalid JSON value type", 0);
				}
			}

			if (jsonType == null)
				jsonType = "string";

			if (! jsonType.equals("string")       &&
			    ! jsonType.equals("string-array") &&
			    ! jsonType.equals("int")          &&
			    ! jsonType.equals("long")         &&
			    ! jsonType.equals("boolean")         )
				throw new ParseException("Invalid JSON value type, must be \"string\", \"string-array\", \"int\", \"long\" or \"boolean\"", 0);
			
			boolean langTagged = false;
			
			if (map.containsKey("langTag")) {
				
				try {
					langTagged = (Boolean)map.get("langTag");
					
				} catch (Exception e) {
					
					throw new ParseException("Invalid language tag option, must be boolean true or false", 0);
				}
			}
			
			return new Directive(ldapAttrName, ldapAttrType, jsonType, langTagged);
		}
		
		
		
		/**
		 * Parses a map of attribute mapping directives from the 
		 * specified map representation.
		 * 
		 * <p>Example map representation (as JSON object);
		 * 
		 * <pre>
		 * {
		 *   "email"   : { "ldapAttr" : "mail" },
		 *   "age"     : { "ldapAttr" : "personAge", "jsonType" : "int" },
		 *   "active"  : { "ldapAttr" : "accountActive", "jsonType" : "boolean" },
		 *   "updated" : { "ldapAttr" : "updateTime", "ldapType" : "time", "jsonType" : "long" },
		 *   ...
		 * }
		 * </pre>
		 *
		 * @param map The map to parse. Must not be {@code null}.
		 * 
		 * @return The map of attribute mapping directives where the
		 *         keys are the JSON object member names.
		 * 
		 * @throws ParseException If parsing of the map failed.
		 */

		public static Map<String,Directive> parseDirectives(final Map<String,Object> map)
			throws ParseException {
			
			Map<String,Directive> directives = new HashMap<String, Directive>();
			
			for (Map.Entry<String,Object> mapping: map.entrySet()) {

				String jsonObjectMemberName = mapping.getKey();
				
				if (mapping.getValue() == null)
					throw new ParseException("Missing mapping directive for \"" + jsonObjectMemberName + "\"", 0);
				
				Directive dir;

				try {
					@SuppressWarnings("unchecked")
					Map<String,Object> dirMap = (Map<String,Object>)mapping.getValue();
					 dir = parse(dirMap);

				} catch (Exception e) {

					throw new ParseException("Invalid mapping directive for \"" + jsonObjectMemberName + "\": " + 
						e.getMessage(), 0);
				}
				
				directives.put(jsonObjectMemberName, dir);
			}
			
			return directives;
		}
	}


	/**
	 * The attribute transformation map.
	 */
	private Map<String,Directive> transformMap;


	/**
	 * The derived names of the source LDAP attributes.
	 */
	private final String[] ldapAttributes;


	/**
	 * Creates a new LDAP attribute mapper.
	 *
	 * @param map The attribute transformation map. Must not be
	 *            {@code null}.
	 *
	 * @throws IllegalArgumentException If the attribute transformation map 
	 *                                  is not valid.
	 */
	public AttributeMapper(final Map<String,Object> map) {

		if (map == null)
			throw new IllegalArgumentException("The attribute transformation map must not be null");
		
		try {
			this.transformMap = Directive.parseDirectives(map);
			
		} catch (ParseException e) {
			
			throw new IllegalArgumentException("The attribute transformation map is invalid: " + e.getMessage(), e);
		}

		
		ldapAttributes = new String[transformMap.size()];
		
		int i=0;
		
		for (Directive dir: transformMap.values()) {
			ldapAttributes[i++] = dir.ldapAttributeName();
		}
	}


	/**
	 * Gets the names of the LDAP attributes in the transformation map.
	 *
	 * @return The names of LDAP attributes in the transformation map, 
	 *         empty array if none.
	 */
	public String[] getLDAPAttributeNames() {

		return ldapAttributes;
	}


	/**
	 * Gets the names of the LDAP attributes in the transformation map that
	 * match the specified JSON object keys.
	 *
	 * @param jsonObjectKey The names of the JSON object keys. Must not be 
	 *                      {@code null}.
	 *
	 * @return The names of the matching LDAP attributes, as a unmodifiable
	 *         list, empty list if none.
	 */
	public List<String> getLDAPAttributeNames(final List<String> jsonObjectKey) {

		List<String> list = new ArrayList<String>(jsonObjectKey.size());

		for (String key: jsonObjectKey) {

			String ldapAttrName = getLDAPAttributeName(key);

			if (ldapAttrName != null)
				list.add(ldapAttrName);
		}

		return Collections.unmodifiableList(list);
	}


	/**
	 * Gets the name of the source LDAP attribute for the specified target
	 * JSON object attribute name.
	 *
	 * @param targetName The target JSON object attribute name. Must not be 
	 *                   {@code null}.
	 *
	 * @return The source LDAP attribute name, {@code null} if not found.
	 */
	public String getLDAPAttributeName(final String targetName) {

		Directive dir = transformMap.get(targetName);
		
		if (dir == null)
			return null;
		
		return dir.ldapAttributeName();
	}


	/**
	 * Transforms the specified LDAP entry. Any transformation exceptions 
	 * are silently ignored.
	 *
	 * @param ldapEntry The LDAP entry to transform. Must not be 
	 *                  {@code null}.
	 *
	 * @return The resulting JSON object.
	 */
	@SuppressWarnings("unchecked")
	public JSONObject transform(final Entry ldapEntry) {

		JSONObject out = new JSONObject();

		for (Map.Entry<String,Directive> mapping: transformMap.entrySet()) {

			try {
				final Directive dir = mapping.getValue();
				
				if (! ldapEntry.hasAttribute(dir.ldapAttributeName()) && ! dir.langTagged())
					continue;
				
				Object jsonValue = null;

				if (dir.jsonValueType().equals("string")) {

					if (dir.langTagged()) {
						// Expect optional lang tag
						for (Attribute a: ldapEntry.getAttributesWithOptions(dir.ldapAttributeName(), null)) {
							
							LangTag langTag = null;
							
							for(String opt: a.getOptions()) {
								
								if (opt.startsWith("lang-")) {
									langTag = LangTag.parse(opt.substring("lang-".length()));
								}
							}
							
							if (langTag != null) {
								out.put(mapping.getKey() + "#" + langTag.toString(), a.getValue());
							} else {
								out.put(mapping.getKey(), a.getValue());
							}
						}
						
					} else {
						// No lang tag
						jsonValue = ldapEntry.getAttributeValue(dir.ldapAttributeName());
					}
					
				} else if (dir.jsonValueType().equals("string-array")) {

					String[] array = ldapEntry.getAttributeValues(dir.ldapAttributeName());

					if (array != null)
						jsonValue = Arrays.asList(array);
					
				} else if (dir.jsonValueType().equals("boolean")) {

					jsonValue = ldapEntry.getAttributeValueAsBoolean(dir.ldapAttributeName());
					
				} else if (dir.jsonValueType().equals("int")) {

					jsonValue = ldapEntry.getAttributeValueAsInteger(dir.ldapAttributeName());
					
				} else if (dir.jsonValueType().equals("long")) {
					
					if (dir.ldapAttributeType().equals("time")) {
						
						String t= ldapEntry.getAttributeValue(dir.ldapAttributeName());
						jsonValue = StaticUtils.decodeGeneralizedTime(t).getTime() / 1000;
						
					} else {
						
						jsonValue = ldapEntry.getAttributeValueAsLong(dir.ldapAttributeName());
					}
				} 

				if (jsonValue != null)
					out.put(mapping.getKey(), jsonValue);

			} catch (Exception e) {

				// ignore
			}
		}

		return out;
	}


	/**
	 * Transforms the specified {@code java.util.Map} representation of an
	 * LDAP entry. Any transformation exceptions are silently ignored.
	 * 
	 * <p>Note: Language tags are not handled by this method.
	 *
	 * @param in The input map, as returned by {@link JSONResultFormatter},
	 *           to transform. Must not be {@code null}.
	 *
	 * @return The resulting JSON object.
	 */
	@SuppressWarnings("unchecked")
	public JSONObject transform(final Map<String,Object> in) {

		JSONObject out = new JSONObject();

		for (Map.Entry<String,Directive> mapping: transformMap.entrySet()) {

			try {
				final Directive dir = mapping.getValue();

				List<Object> valuesIn = (List<Object>)in.get(dir.ldapAttributeName());
				
				if (valuesIn == null)
					continue;

				Object jsonValue = null;

				if (dir.jsonValueType().equals("string")) {

					jsonValue = (String)valuesIn.get(0);
					
				} else if (dir.jsonValueType().equals("string-array")) {

					jsonValue = valuesIn;
					
				} else if (dir.jsonValueType().equals("boolean")) {

					jsonValue = Boolean.parseBoolean((String)valuesIn.get(0));
					
				} else if (dir.jsonValueType().equals("int")) {

					jsonValue =  Integer.parseInt((String)valuesIn.get(0));
					
				} else if (dir.jsonValueType().equals("long")) {

					if (dir.ldapAttributeType().equals("time")) {
						
						String t = (String)valuesIn.get(0);
						jsonValue = StaticUtils.decodeGeneralizedTime(t).getTime() / 1000;
						
					} else {
						
						jsonValue = Long.parseLong((String)valuesIn.get(0));
					}
				} 

				// Output
				if (jsonValue != null)
					out.put(mapping.getKey(), jsonValue);

			} catch (Exception e) {

				// skip
			} 
		}

		return out;
	}


	/**
	 * Performs a reverse transformation of the specified JSON object
	 * representation of an LDAP entry. Any transformation exceptions are 
	 * silently ignored.
	 *
	 * @param jsonObject The JSON object to transform in reverse. Must not 
	 *                   be {@code null}.
	 *
	 * @return The resulting LDAP attributes for the entry.
	 */
	@SuppressWarnings("unchecked")
	public List<Attribute> reverseTransform(final JSONObject jsonObject) {

		List<Attribute> ldapAttrs = new ArrayList<Attribute>(jsonObject.size());

		for (Map.Entry<String,Directive> mapping: transformMap.entrySet()) {

			try {
				final Directive dir = mapping.getValue();
				
				if (dir.langTagged() &&
				    dir.jsonValueType().equals("string") && 
				    dir.ldapAttributeType().equals("string")) {
					
					// Extract the LangTags for the key, if any
					Map<LangTag,Object> langTaggedEntries = LangTagUtil.find(mapping.getKey(), jsonObject);
					
					for (Map.Entry<LangTag,Object> entry: langTaggedEntries.entrySet()) {
						
						if (entry.getValue() == null)
							continue;
						
						String ldapAttributeName = dir.ldapAttributeName();
					
						LangTag langTag = entry.getKey();
					
						if (langTag != null)
							ldapAttributeName += ";lang-" + langTag.toString();
						
						ldapAttrs.add(new Attribute(ldapAttributeName, entry.getValue().toString()));
					}
					
					continue;
				}
					
				final Object jsonValue = jsonObject.get(mapping.getKey());
				
				if (jsonValue instanceof List) {

					List<Object> jsonValues = (List)jsonValue;
					List<String> ldapValues = new ArrayList<String>();

					for (Object v : jsonValues) {

						if (v instanceof Long
						    && dir.jsonValueType().equals("long")
						    && dir.jsonValueType().equals("time")) {

							// Convert Unix time to LDAP generalised time
							long t = (Long) jsonValue;
							ldapValues.add(StaticUtils.encodeGeneralizedTime(new Date(t*1000)));
						} else {
							ldapValues.add(v.toString());
						}
					}

					ldapAttrs.add(new Attribute(dir.ldapAttributeName(), ldapValues));

				} else if (jsonValue instanceof Long
					   && dir.jsonValueType().equals("long")
					   && dir.ldapAttributeType().equals("time")) {

					// Convert Unix time to LDAP generalised time
					long t = (Long)jsonValue;
					ldapAttrs.add(new Attribute(dir.ldapAttributeName(), StaticUtils.encodeGeneralizedTime(new Date(t*1000))));
					
				} else {
					
					ldapAttrs.add(new Attribute(dir.ldapAttributeName(), jsonValue.toString()));
				}

			} catch (Exception e) {
				// skip
			}
		}

		return ldapAttrs;
	}
}