package com.nimbusds.common.servlet;


import java.security.Principal;
import java.security.cert.X509Certificate;

import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;


/**
 * Helper methods for Log4j message generation.
 *
 * @author Vladimir Dzhuvinov
 */
public class ServletLogUtility {


	/**
	 * The logger.
	 */
	private static final Logger log = Logger.getLogger(ServletLogUtility.class);
	
	
	/**
	 * Logs (at INFO level) the method, the client IP address, CORS origin
	 * (if any) and the security client certificate principal (if any) of 
	 * an HTTP request.
	 *
	 * <p>Example:
	 *
	 * <pre>
	 * HTTP POST request from 127.0.0.1
	 * </pre>
	 *
	 * @param httpRequest The HTTP request to log.
	 */
	public static void log(final HttpServletRequest httpRequest) {
		
		if (! log.isInfoEnabled())
			return;
		
		final String hostIp = httpRequest.getRemoteAddr();
		final String method = httpRequest.getMethod();
		final String corsOrigin = String.valueOf(httpRequest.getAttribute("cors.origin"));
		
		
		// Compose log message
		StringBuilder msg = new StringBuilder("HTTP");
		
		if (httpRequest.isSecure())
			msg.append("S");
		
		if (! corsOrigin.equals("null"))
			msg.append(" CORS ");
		else
			msg.append(" ");
		
		msg.append(method + " request from " + hostIp);
		
		if (! corsOrigin.equals("null"))
			msg.append(" origin: " + corsOrigin);
		
		
		X509Certificate[] certs = 
			(X509Certificate[])httpRequest.getAttribute("javax.servlet.request.X509Certificate");
		
		if (certs != null) {
		
			for (int i=0; i < certs.length; i++) {
			
				Principal principal = certs[i].getSubjectX500Principal();
			
				if (principal == null)
					continue;
			
				String principalName = principal.getName();
				
				msg.append(" principal[" + i + "]: " + principalName);
			}
		}
		
		log.info(msg.toString());
	}


	/**
	 * Prevents public instantiation.
	 */
	private ServletLogUtility() { }
}
