package com.nimbusds.common.servlet;


import java.io.InputStream;
import java.io.IOException;

import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;

import org.infinispan.manager.DefaultCacheManager;
import org.infinispan.manager.EmbeddedCacheManager;

import org.apache.log4j.Logger;


/**
 * Configures and launches an Infinispan cache manager at servlet context
 * startup. The cache manager is stopped at servlet context shutdown (if
 * previously launched).
 *
 * <p>The name / path of the Infinispan configuration file is specified in a
 * servlet context init parameter {@code infinispan.configurationFile}.
 *
 * <p>The launched Infinispan cache manager will be exported to the servlet 
 * context as an {@code org.infinispan.manager.EmbeddedCacheManager}, under a 
 * key that bears the interface name.
 *
 * @author Vladimir Dzhuvinov
 */
public class InfinispanLauncher implements ServletContextListener {
	
	
	/**
	 * The Infinispan cache manager.
	 */
	private EmbeddedCacheManager cacheManager;


	/**
	 * Handler for servlet context startup events; configures and launches
	 * an Infinispan cache manager using the configuration file specified 
	 * in the servlet context parameter {@code infinispan.configurationFile}. 
	 *
	 * <p>The configuration file location must be relative to the web 
	 * application directory, e.g. {@code /WEB-INF/infinispan.xml}.
	 *
	 * <p>Exceptions are logged at ERROR level using Log4j.
	 *
	 * @param sce A servlet context event.
	 */
	@Override
	public void contextInitialized(ServletContextEvent sce)
		throws IllegalArgumentException {

		Logger log = Logger.getLogger(InfinispanLauncher.class);

		ServletContext servletContext = sce.getServletContext();

		String configFile = servletContext.getInitParameter("infinispan.configurationFile");
		
		if (configFile == null || configFile.trim().isEmpty()) {

			log.error("Couldn't load Infinispan configuration: Missing servlet context parameter \"infinispan.configurationFile\"");
			return;
		}
		
		InputStream is = servletContext.getResourceAsStream(configFile);
		
		if (is == null) {

			log.error("Couldn't load Infinispan configuration file: " + configFile);
			return;
		}
		
		try {
			cacheManager = new DefaultCacheManager(is);

		} catch (IOException e) {

			log.error("Couldn't start Infinispan cache manager: " + e.getMessage());
			return;
		}

		servletContext.setAttribute("org.infinispan.manager.EmbeddedCacheManager", cacheManager);
	}


	/**
	 * Handler for servlet context shutdown events. Stops the Infinispan
	 * cache manager.
	 *
	 * @param sce A servlet context event.
	 */
	@Override
	public void contextDestroyed(ServletContextEvent sce) {

		if (cacheManager == null)
			return;

		cacheManager.stop();
	}
}
