package com.nimbusds.common.servlet;


import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.IOException;
import java.util.Properties;

import javax.servlet.ServletContext;

import org.apache.log4j.Logger;


/**
 * Servlet resource retriever. Can be used to retrieve the content of a
 * configuration file, as {@code java.io.InputStream}, {@code String} or
 * {@code java.util.Properties}.
 */
public class ResourceRetriever {

	
	/**
	 * Gets the resource stored in the text file which location is 
	 * specified in a {@code web.xml} context parameter.
	 *
	 * @param servletCtx  The servlet context. Must not be {@code null}.
	 * @param paramName   The name of the servlet context init parameter
	 *                    specifying the properties file location (relative 
	 *                    to the web app root). For example, 
	 *                    {@code /WEB-INF/authService.properties}. Must not 
	 *                    be {@code null}.
	 * @param log         To log exceptions at ERROR level. Must not be 
	 *                    {@code null}.
	 *
	 * @return The resource as an input stream.
         *
         * @throws Exception On a missing servlet context parameter or a 
         *                   missing resource.
	 */
	public static InputStream getStream(final ServletContext servletCtx,
		                            final String paramName,
		                            final Logger log)
		throws Exception {

		String file = servletCtx.getInitParameter(paramName);
		
		if (file == null || file.trim().isEmpty()) {
			final String msg = "Missing " + paramName + " context parameter in web.xml";
			log.error(msg);
			throw new Exception(msg);
		}

		InputStream is = servletCtx.getResourceAsStream(file);

		if (is == null) {
			final String msg = "Missing file or invalid path: " + file;
			log.error(msg);
			throw new Exception(msg);
		}

		return is;
	}


	/**
	 * Gets the resource stored in the text file which location is 
	 * specified in a {@code web.xml} context parameter.
	 *
	 * @param servletCtx  The servlet context. Must not be {@code null}.
	 * @param paramName   The name of the servlet context init parameter
	 *                    specifying the properties file location (relative 
	 *                    to the web app root). For example, 
	 *                    {@code /WEB-INF/authService.properties}. Must not 
	 *                    be {@code null}.
	 * @param log         To log exceptions at ERROR level. Must not be 
	 *                    {@code null}.
	 *
	 * @return The resource as a string.
         *
         * @throws Exception On a missing servlet context parameter, missing or
         *                   bad resource file.
	 */
	public static String getString(final ServletContext servletCtx,
		                       final String paramName,
		                       final Logger log)
		throws Exception {

		InputStream is = getStream(servletCtx, paramName, log);

		StringBuilder sb = new StringBuilder();

		try {
			BufferedReader reader = new BufferedReader(new InputStreamReader(is));

			String line;

			while ((line = reader.readLine()) != null) {
				sb.append(line);
				sb.append(System.getProperty("line.separator"));
			}

			reader.close();

		} catch (IOException e) {
			final String msg = "Couldn't read file: " + e.getMessage();
			log.error(msg);
			throw new Exception(msg, e);
		}

		return sb.toString();
	}


	/**
         * Gets the java.util.Properties stored in the text file which location
	 * is specified in a {@code web.xml} context parameter.
         *
	 * @param servletCtx  The servlet context. Must not be {@code null}.
	 * @param paramName   The name of the servlet context init parameter
	 *                    specifying the properties file location (relative 
	 *                    to the web app root). For example, 
	 *                    {@code /WEB-INF/authService.properties}. Must not 
	 *                    be {@code null}.
	 * @param log         To log exceptions at ERROR level. Must not be 
	 *                    {@code null}.
	 *
	 * @return The properties.
         *
         * @throws Exception On a missing servlet context parameter, missing or 
	 *                   bad properties file.
         */
	public static Properties getProperties(final ServletContext servletCtx,
					       final String paramName,
					       final Logger log)
		throws Exception {
		
		String file = servletCtx.getInitParameter(paramName);

		InputStream is = getStream(servletCtx, paramName, log);

		
                Properties props = new Properties();
		
		// Read the properties
		try {
			props.load(is);
			
		} catch (IOException e) {
			final String msg = "Couldn't read properties file: " + e.getMessage();
			log.error(msg);
			throw new Exception(msg, e);
			
		} catch (IllegalArgumentException e) {
			final String msg = "Bad properties file: " + e.getMessage();
			log.error(msg);
			throw new Exception(msg, e);
		}
		
		log.info("Loaded properties file " + file);
		
		return props;
	}
	
	
	/**
	 * Prevents public instantiation.
	 */
	private ResourceRetriever() { }
}
