package com.nimbusds.common.oauth2;


import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import org.apache.commons.lang3.StringUtils;

import net.minidev.json.JSONObject;

import com.nimbusds.oauth2.sdk.ParseException;
import com.nimbusds.oauth2.sdk.token.BearerAccessToken;
import com.nimbusds.oauth2.sdk.token.BearerTokenError;


/**
 * Basic access token validator, intended for JAX-RS web applications.
 */
public class BasicAccessTokenValidator {


	/**
	 * The access token, {@code null} access to the web API is disabled.
	 */
	private final BearerAccessToken accessToken;


	/**
	 * Error response: Missing OAuth 2.0 Bearer access token.
	 *
	 * <p>JSON object:
	 *
	 * <pre>
	 * {
	 *   "error"             : "missing_token",
	 *   "error_description" : "Unauthorized: Missing Bearer access token"
	 * }
	 * </pre>
	 */
	public static final WebApplicationException MISSING_BEARER_TOKEN;


	/**
	 * Error response: Invalid OAuth 2.0 Bearer access token.
	 *
	 * <p>JSON object:
	 *
	 * <pre>
	 * {
	 *   "error"             : "invalid_token",
	 *   "error_description" : "Unauthorized: Invalid Bearer access token"
	 * }
	 * </pre>
	 */
	public static final WebApplicationException INVALID_BEARER_TOKEN;


	/**
	 * Error response: Web API disabled.
	 *
	 * <p>JSON object:
	 *
	 * <pre>
	 * {
	 *   "error"             : "web_api_disabled",
	 *   "error_description" : "Forbidden: Web API disabled"
	 * }
	 * </pre>
	 */
	public static final WebApplicationException WEB_API_DISABLED;


	static {
		JSONObject o = new JSONObject();
		o.put("error", "missing_token");
		o.put("error_description", "Unauthorized: Missing Bearer access token");
		MISSING_BEARER_TOKEN = new WebApplicationException(
			Response.status(BearerTokenError.MISSING_TOKEN.getHTTPStatusCode()).
				header("WWW-Authenticate", BearerTokenError.MISSING_TOKEN.toWWWAuthenticateHeader()).
				entity(o.toJSONString()).
				type(MediaType.APPLICATION_JSON).
				build());

		o = new JSONObject();
		o.put("error", BearerTokenError.INVALID_TOKEN.getCode());
		o.put("error_description", "Unauthorized: Invalid Bearer access token");
		INVALID_BEARER_TOKEN = new WebApplicationException(
			Response.status(BearerTokenError.INVALID_TOKEN.getHTTPStatusCode()).
				header("WWW-Authenticate", BearerTokenError.INVALID_TOKEN.toWWWAuthenticateHeader()).
				entity(o.toJSONString()).
				type(MediaType.APPLICATION_JSON).
				build());

		o = new JSONObject();
		o.put("error", "web_api_disabled");
		o.put("error_description", "Forbidden: Web API disabled");
		WEB_API_DISABLED = new WebApplicationException(
			Response.status(Response.Status.FORBIDDEN).
				entity(o.toJSONString()).
				type(MediaType.APPLICATION_JSON).
				build());
	}


	/**
	 * Creates a new basic access token validator.
	 *
	 * @param accessToken The Bearer access token. If {@code null} access
	 *                    to the web API will be disabled.
	 */
	public BasicAccessTokenValidator(final BearerAccessToken accessToken) {

		this.accessToken = accessToken;
	}


	/**
	 * Returns the Bearer access token.
	 *
	 * @return The Bearer access token, {@code null} access to the web API
	 *         is disabled.
	 */
	public BearerAccessToken getAccessToken() {

		return accessToken;
	}


	/**
	 * Validates a Bearer access token passed in the specified HTTP
	 * Authorization header value.
	 *
	 * @param authzHeader The HTTP Authorization header value, {@code null}
	 *                    if not specified.
	 *
	 * @throws WebApplicationException If the header value is {@code null},
	 *                                 the web API is disabled, or the
	 *                                 Bearer access token is missing or
	 *                                 invalid.
	 */
	public void validateBearerAccessToken(final String authzHeader)
		throws WebApplicationException {

		if (StringUtils.isBlank(authzHeader))
			throw MISSING_BEARER_TOKEN;

		BearerAccessToken token;

		try {
			token = BearerAccessToken.parse(authzHeader);

		} catch (ParseException e) {

			throw MISSING_BEARER_TOKEN;
		}

		// Web API disabled?
		if (accessToken == null)
			throw WEB_API_DISABLED;

		// Check token
		if (!token.equals(accessToken))
			throw INVALID_BEARER_TOKEN;
	}
}

