package com.nimbusds.common.metrics;


import java.util.Map;

import com.codahale.metrics.Metric;
import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.MetricSet;
import com.codahale.metrics.health.HealthCheck;
import com.codahale.metrics.health.HealthCheckRegistry;


/**
 * Shared Dropwizard Metrics registries.
 *
 * @see com.nimbusds.common.servlet.MetricsLauncher
 */
public class MetricsRegistries {


	/**
	 * The metrics registry.
	 */
	private static MetricRegistry metricRegistry;


	/**
	 * The health check registry.
	 */
	private static HealthCheckRegistry healthCheckRegistry;


	/**
	 * Returns the singleton shared registry for metric instances.
	 *
	 * @return The registry.
	 */
	public static MetricRegistry getMetricRegistry() {

		if (metricRegistry != null) {
			return metricRegistry;
		}

		metricRegistry = new MetricRegistry();
		return metricRegistry;
	}


	/**
	 * Returns the singleton shared registry for health check instances.
	 *
	 * @return The registry.
	 */
	public static HealthCheckRegistry getHealthCheckRegistry() {

		if (healthCheckRegistry != null) {
			return healthCheckRegistry;
		}

		healthCheckRegistry = new HealthCheckRegistry();
		return healthCheckRegistry;
	}


	/**
	 * Registers a metric set, while guarding against name collision.
	 *
	 * @param metricSet The metric set to register. If {@code null} the
	 *                  method will return immediately.
	 */
	public static void register(final MetricSet metricSet) {

		if (metricSet == null)
			return;

		for (Map.Entry<String,Metric> entry: metricSet.getMetrics().entrySet())
			register(entry.getKey(), entry.getValue());
	}


	/**
	 * Registers a metric, while guarding against name collision.
	 *
	 * @param name   The metric name. If {@code null} the method will
	 *               return immediately.
	 * @param metric The metric. If {@code null} the method will return
	 *               immediately.
	 */
	public static void register(final String name, final Metric metric) {

		if (name == null || metric == null)
			return;

		if (getMetricRegistry().getNames().contains(name))
			return; // skip

		getMetricRegistry().register(name, metric);
	}


	/**
	 * Registers a health check, while guarding against name collision.
	 *
	 * @param name  The health check name. If {@code null} the method will
	 *              return immediately.
	 * @param check The health check. If {@code null} the method will
	 *              return immediately.
	 */
	public static void register(final String name, final HealthCheck check) {

		if (name == null || check == null)
			return;

		if (getHealthCheckRegistry().getNames().contains(name))
			return; // skip

		getHealthCheckRegistry().register(name, check);
	}


	/**
	 * Prevents public instantiation.
	 */
	private MetricsRegistries() { }
}
