package com.nimbusds.common.servlet;


import java.util.Properties;
import java.util.concurrent.TimeUnit;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;

import com.codahale.metrics.JmxReporter;
import com.codahale.metrics.graphite.Graphite;
import com.codahale.metrics.graphite.GraphiteReporter;
import com.codahale.metrics.graphite.GraphiteSender;
import com.codahale.metrics.graphite.PickledGraphite;
import com.nimbusds.common.config.ConfigurationException;
import com.nimbusds.common.config.MonitorConfiguration;
import com.nimbusds.common.monitor.MonitorRegistries;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;


/**
 * Monitor launcher. Exports the shared
 * {@link com.nimbusds.common.monitor.MonitorRegistries} into the servlet
 * context.
 */
public class MonitorLauncher implements ServletContextListener {


	/**
	 * The name of the servlet context parameter that specifies the
	 * configuration file location.
	 */
	public static final String CONFIG_CTX_PARAMETER_NAME = "monitor.configurationFile";


	/**
	 * The JMX reporter.
	 */
	protected JmxReporter jmxReporter;


	/**
	 * The Graphite reporter.
	 */
	protected GraphiteReporter graphiteReporter;


	@Override
	public void contextInitialized(final ServletContextEvent sce) {

		sce.getServletContext().setAttribute(
			"com.codahale.metrics.servlets.MetricsServlet.registry",
			MonitorRegistries.getMetricRegistry());

		sce.getServletContext().setAttribute("com.codahale.metrics.servlets.HealthCheckServlet.registry",
			MonitorRegistries.getHealthCheckRegistry());

		Logger log = LogManager.getLogger("MAIN");

		Properties props;

		try {
			props = ResourceRetriever.getProperties(
				sce.getServletContext(),
				CONFIG_CTX_PARAMETER_NAME,
				log);

		} catch (Exception e) {
			log.error(e.getMessage(), e);
			throw new RuntimeException(e.getMessage(), e);
		}

		final MonitorConfiguration config;

		try {
			config = new MonitorConfiguration(props);

		} catch (ConfigurationException e) {
			log.error(e.getMessage(), e);
			throw e;
		}

		if (config.enableJMX) {
			jmxReporter = JmxReporter.forRegistry(MonitorRegistries.getMetricRegistry()).build();
			jmxReporter.start();
			log.info("[CM7100] Started JMX metrics reporter");
		} else {
			log.info("[CM7101] JMX metrics reporting disabled");
		}


		if (config.graphite.enable) {
			final GraphiteSender graphite;
			if (config.graphite.batchSize > 0) {
				graphite = new PickledGraphite(config.graphite.host, config.graphite.port, config.graphite.batchSize);
			} else {
				graphite = new Graphite(config.graphite.host, config.graphite.port);
			}
			GraphiteReporter.Builder builder = GraphiteReporter.forRegistry(MonitorRegistries.getMetricRegistry());

			if (config.graphite.prefix != null && ! config.graphite.prefix.isEmpty()) {
				builder = builder.prefixedWith(config.graphite.prefix);
			}

			graphiteReporter = builder.convertRatesTo(config.graphite.ratesTimeUnit)
				.convertDurationsTo(config.graphite.durationsTimeUnit)
				.filter(config.graphite.filter)
				.build(graphite);

			graphiteReporter.start(config.graphite.reportInterval, TimeUnit.SECONDS);
			log.info("[CM7102] Started Graphite metrics reporting");
		} else {
			log.info("[CM7103] Graphite metrics reporting disabled");
		}
	}


	@Override
	public void contextDestroyed(final ServletContextEvent sce) {

		if (jmxReporter != null) {
			jmxReporter.stop();
		}

		if (graphiteReporter != null) {
			graphiteReporter.stop();
		}
	}
}
