package com.nimbusds.common.servlet;


import java.io.IOException;
import java.util.Properties;
import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.codahale.metrics.servlets.AdminServlet;
import com.nimbusds.common.config.ConfigurationException;
import com.nimbusds.common.oauth2.BasicAccessTokenValidator;
import com.nimbusds.oauth2.sdk.token.BearerAccessToken;
import com.thetransactioncompany.util.PropertyParseException;
import com.thetransactioncompany.util.PropertyRetriever;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;


/**
 * Monitor servlet for exposing Dropwizard metrics and health checks, requires
 * an OAuth 2.0 bearer token for access.
 *
 * <p>The token is defined in a {@link Configuration#API_TOKEN_PROPERTY_NAME
 * Java property} obtained from 1) system properties or from 2) a properties
 * file specified by
 * {@link com.nimbusds.common.servlet.MonitorLauncher#CONFIG_CTX_PARAMETER_NAME
 * servlet context parameter}.
 */
public class MonitorServlet extends AdminServlet {
	
	
	/**
	 * The monitor servlet access token configuration.
	 */
	public static class Configuration {
		
		
		/**
		 * The property name for the API access token.
		 */
		public static final String API_TOKEN_PROPERTY_NAME = "monitor.apiAccessToken";
		
		
		/**
		 * The property name for the secondary API access token.
		 */
		public static final String SECONDARY_API_TOKEN_PROPERTY_NAME = "monitor.secondaryAPIAccessToken";
		
		
		/**
		 * The min accepted web access token character length.
		 */
		public static final int MIN_TOKEN_LENGTH = 32;
		
		
		/**
		 * The monitor API access token.
		 */
		public final BearerAccessToken apiAccessToken;
		
		
		/**
		 * Optional secondary monitor API access token.
		 */
		public final BearerAccessToken secondaryAPIAccessToken;
		
		
		/**
		 * Creates a new monitor servlet access token configuration.
		 *
		 * @param properties The configuration properties, can be
		 *                   overridden with system properties.
		 *
		 * @throws ConfigurationException On a invalid configuration
		 *                                property.
		 */
		public Configuration(final Properties properties)
			throws ConfigurationException {
			
			if (System.getProperties().containsKey(API_TOKEN_PROPERTY_NAME)) {
				// Override with system property
				properties.setProperty(API_TOKEN_PROPERTY_NAME, System.getProperty(API_TOKEN_PROPERTY_NAME));
			}
			
			if (System.getProperties().containsKey(SECONDARY_API_TOKEN_PROPERTY_NAME)) {
				// Override
				properties.setProperty(SECONDARY_API_TOKEN_PROPERTY_NAME, System.getProperty(SECONDARY_API_TOKEN_PROPERTY_NAME));
			}
			
			
			PropertyRetriever pr = new PropertyRetriever(properties);
			
			try {
				String s = pr.getString(API_TOKEN_PROPERTY_NAME);
				
				if (s.trim().length() < MIN_TOKEN_LENGTH) {
					String msg = "The monitor web API access token must be at least " + MIN_TOKEN_LENGTH + " characters long";
					throw new PropertyParseException(msg, API_TOKEN_PROPERTY_NAME, s);
				}
				
				apiAccessToken = new BearerAccessToken(s);
				
				s = pr.getOptString(SECONDARY_API_TOKEN_PROPERTY_NAME, null);
				
				if (s != null && s.trim().length() < MIN_TOKEN_LENGTH) {
					String msg = "The secondary monitor web API access token must be at least " + MIN_TOKEN_LENGTH + " characters long";
					throw new PropertyParseException(msg, SECONDARY_API_TOKEN_PROPERTY_NAME, s);
				}
				
				if (s != null) {
					secondaryAPIAccessToken = new BearerAccessToken(s);
				} else {
					secondaryAPIAccessToken = null;
				}
				
			} catch (PropertyParseException e) {
				throw new ConfigurationException(e.getMessage() + ": Property: " + e.getPropertyKey(), e);
			}
		}
	}


	/**
	 * The access token validator.
	 */
	protected BasicAccessTokenValidator tokenValidator;
	
	
	/**
	 * Creates a new basic access token validator.
	 *
	 * @param config The servlet configuration, used to retrieve the
	 *               monitor configuration properties.
	 *
	 * @return The basic access token validator.
	 *
	 * @throws ServletException If a configuration property is missing or
	 *                          invalid.
	 */
	static BasicAccessTokenValidator createBasicAccessTokenValidator(final ServletConfig config)
		throws ServletException {
		
		Logger log = LogManager.getLogger("MAIN");
		
		Configuration monitorConfig;
		
		try {
			Properties props = ResourceRetriever.getProperties(
				config.getServletContext(),
				MonitorLauncher.CONFIG_CTX_PARAMETER_NAME,
				log);
			
			monitorConfig = new Configuration(props);
			
		} catch (Exception e) {
			log.error(e.getMessage(), e);
			throw new ServletException(e.getMessage(), e);
		}
		
		if (monitorConfig.secondaryAPIAccessToken != null) {
			return new BasicAccessTokenValidator(monitorConfig.apiAccessToken, monitorConfig.secondaryAPIAccessToken);
		} else {
			return new BasicAccessTokenValidator(monitorConfig.apiAccessToken);
		}
	}


	@Override
	public void init(final ServletConfig config)
		throws ServletException {

		super.init(config);
		
		tokenValidator = createBasicAccessTokenValidator(config);
	}


	@Override
	protected void doGet(final HttpServletRequest req,
			     final HttpServletResponse resp)
		throws ServletException, IOException {

		if (! tokenValidator.validateBearerAccessToken(req, resp)) {
			return; // invalid or missing token
		}

		super.doGet(req, resp);
	}


	@Override
	protected void service(final HttpServletRequest req,
			       final HttpServletResponse resp)
		throws ServletException, IOException {

		if (! tokenValidator.validateBearerAccessToken(req, resp)) {
			return; // invalid or missing token
		}

		super.service(req, resp);
	}
}
